<?php
/**
 * Content Watching Page (Pay-per-view)
 */

// Include configuration
require_once 'includes/config.php';

// Include required classes
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/content.php';
require_once 'includes/payment.php';

// Initialize classes
$auth = new Auth();
$contentManager = new Content();
$payment = new Payment();

// Check if content ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: index.php');
    exit;
}

$contentId = (int)$_GET['id'];

// Get content details
$contentItem = $contentManager->getContentById($contentId);

// If content doesn't exist or is not published, redirect to home
if (!$contentItem || $contentItem['status'] !== 'published') {
    header('Location: index.php');
    exit;
}

// Get current user if logged in
$currentUser = $auth->getCurrentUser();
$userId = $currentUser ? $currentUser['id'] : null;

// Check if this is a free content or if user has already purchased
$isFree = ($contentItem['price'] <= 0);
$hasPurchased = $userId ? $payment->hasContentAccess($userId, $contentId) : false;
$canWatch = $isFree || $hasPurchased;

// If user is trying to watch content without purchase, show payment page
$showPayment = !$canWatch;

// If user can watch, log the view
if ($canWatch) {
    $contentManager->logContentView($contentId, $userId);
}

// Get related content
$relatedContent = $contentManager->getRelatedContent($contentId, 4);

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "languages/{$lang}.php";

// Get page title
$pageTitle = htmlspecialchars($contentItem['title']) . ' - ' . SITE_NAME;

// Include header
include 'includes/header.php';
?>

<section class="content-section py-5">
    <div class="container">
        <div class="row">
            <div class="col-lg-8">
                <?php if ($canWatch): ?>
                    <!-- Video Player -->
                    <div class="video-container mb-4">
                        <?php if ($contentItem['content_type'] === 'video'): ?>
                            <video controls class="w-100" poster="<?php echo htmlspecialchars($contentItem['thumbnail']); ?>">
                                <source src="<?php echo htmlspecialchars($contentItem['file_url']); ?>" type="video/mp4">
                                <?php echo $lang_data['browser_not_support']; ?>
                            </video>
                        <?php elseif ($contentItem['content_type'] === 'audio'): ?>
                            <div class="audio-player text-center py-4" 
                                 style="background: url('<?php echo htmlspecialchars($contentItem['thumbnail']); ?>') center/cover no-repeat;">
                                <audio controls class="w-75">
                                    <source src="<?php echo htmlspecialchars($contentItem['file_url']); ?>" type="audio/mpeg">
                                    <?php echo $lang_data['browser_not_support']; ?>
                                </audio>
                            </div>
                        <?php else: ?>
                            <div class="embed-responsive embed-responsive-16by9">
                                <iframe class="embed-responsive-item" 
                                        src="<?php echo htmlspecialchars($contentItem['embed_url']); ?>" 
                                        allowfullscreen></iframe>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php else: ?>
                    <!-- Payment Required Box -->
                    <div class="payment-required-box text-center p-5 bg-light mb-4">
                        <img src="<?php echo htmlspecialchars($contentItem['thumbnail']); ?>" alt="<?php echo htmlspecialchars($contentItem['title']); ?>" 
                             class="img-fluid rounded mb-4" style="max-height: 300px;">
                        <h3><?php echo $lang_data['premium_content']; ?></h3>
                        <p><?php echo $lang_data['purchase_to_watch']; ?></p>
                        <div class="price-badge my-3">
                            <h4><?php echo number_format($contentItem['price']) . ' ' . CURRENCY; ?></h4>
                        </div>
                        
                        <?php if (!$auth->isLoggedIn()): ?>
                            <div class="alert alert-info">
                                <?php echo $lang_data['login_to_purchase']; ?>
                            </div>
                            <div class="d-grid gap-2 col-6 mx-auto">
                                <a href="login.php?redirect=watch.php?id=<?php echo $contentId; ?>" class="btn btn-primary">
                                    <?php echo $lang_data['login']; ?>
                                </a>
                                <a href="register.php?redirect=watch.php?id=<?php echo $contentId; ?>" class="btn btn-outline-primary">
                                    <?php echo $lang_data['register']; ?>
                                </a>
                            </div>
                        <?php else: ?>
                            <form action="process-content-payment.php" method="post" class="mt-4">
                                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                                <input type="hidden" name="content_id" value="<?php echo $contentId; ?>">
                                <input type="hidden" name="amount" value="<?php echo $contentItem['price']; ?>">
                                
                                <div class="payment-options mb-4">
                                    <div class="payment-option d-flex align-items-center p-3 mb-2 bg-white rounded border">
                                        <input type="radio" name="payment_method" id="flooz" value="flooz" checked required>
                                        <label for="flooz" class="ms-2 d-flex align-items-center flex-grow-1">
                                            <img src="assets/images/flooz.png" alt="Flooz" class="me-2" style="height: 30px;">
                                            Flooz Mobile Money
                                        </label>
                                    </div>
                                    <div class="payment-option d-flex align-items-center p-3 bg-white rounded border">
                                        <input type="radio" name="payment_method" id="tmoney" value="tmoney">
                                        <label for="tmoney" class="ms-2 d-flex align-items-center flex-grow-1">
                                            <img src="assets/images/tmoney.png" alt="T-Money" class="me-2" style="height: 30px;">
                                            T-Money
                                        </label>
                                    </div>
                                </div>
                                
                                <button type="submit" class="btn btn-primary btn-lg">
                                    <?php echo $lang_data['purchase_now']; ?>
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
                
                <!-- Content Details -->
                <div class="content-details mb-4">
                    <h1><?php echo htmlspecialchars($contentItem['title']); ?></h1>
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <div class="meta-info text-muted">
                            <span><i class="far fa-calendar-alt me-1"></i> <?php echo date('F j, Y', strtotime($contentItem['created_at'])); ?></span>
                            <span class="ms-3"><i class="far fa-eye me-1"></i> <?php echo number_format($contentItem['view_count']); ?> <?php echo $lang_data['views']; ?></span>
                            <?php if ($contentItem['content_type'] === 'video'): ?>
                                <span class="ms-3"><i class="far fa-clock me-1"></i> <?php echo $contentItem['duration']; ?> <?php echo $lang_data['minutes']; ?></span>
                            <?php endif; ?>
                        </div>
                        
                        <div class="share-buttons">
                            <button class="btn btn-sm btn-outline-primary" onclick="shareContent('facebook')">
                                <i class="fab fa-facebook-f"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-info" onclick="shareContent('twitter')">
                                <i class="fab fa-twitter"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-success" onclick="shareContent('whatsapp')">
                                <i class="fab fa-whatsapp"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div class="description">
                        <?php echo nl2br(htmlspecialchars($contentItem['description'])); ?>
                    </div>
                    
                    <?php if (!empty($contentItem['tags'])): ?>
                        <div class="tags mt-3">
                            <?php 
                            $tags = explode(',', $contentItem['tags']);
                            foreach ($tags as $tag): 
                                $tag = trim($tag);
                                if (!empty($tag)):
                            ?>
                                <a href="content.php?tag=<?php echo urlencode($tag); ?>" class="badge bg-secondary text-decoration-none">
                                    <?php echo htmlspecialchars($tag); ?>
                                </a>
                            <?php 
                                endif;
                            endforeach; 
                            ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Comments Section -->
                <?php if ($canWatch): ?>
                    <div class="comments-section">
                        <h3><?php echo $lang_data['comments']; ?></h3>
                        
                        <?php if ($auth->isLoggedIn()): ?>
                            <form action="process-comment.php" method="post" class="mb-4">
                                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                                <input type="hidden" name="content_id" value="<?php echo $contentId; ?>">
                                
                                <div class="mb-3">
                                    <textarea class="form-control" name="comment" rows="3" placeholder="<?php echo $lang_data['write_comment']; ?>" required></textarea>
                                </div>
                                <button type="submit" class="btn btn-primary">
                                    <?php echo $lang_data['post_comment']; ?>
                                </button>
                            </form>
                        <?php else: ?>
                            <div class="alert alert-info">
                                <a href="login.php?redirect=watch.php?id=<?php echo $contentId; ?>"><?php echo $lang_data['login']; ?></a> 
                                <?php echo $lang_data['to_post_comment']; ?>
                            </div>
                        <?php endif; ?>
                        
                        <div class="comments-list">
                            <?php
                            // Get comments for this content
                            $query = "SELECT c.*, u.name as user_name, u.profile_image
                                      FROM comments c
                                      JOIN users u ON c.user_id = u.id
                                      WHERE c.content_id = ? AND c.status = 'approved'
                                      ORDER BY c.created_at DESC";
                            
                            $comments = $db->getRows($query, [$contentId]);
                            
                            if (empty($comments)):
                            ?>
                                <div class="text-center text-muted my-5">
                                    <p><?php echo $lang_data['no_comments']; ?></p>
                                </div>
                            <?php else: ?>
                                <?php foreach ($comments as $comment): ?>
                                    <div class="comment d-flex mb-4">
                                        <div class="comment-avatar me-3">
                                            <?php if (!empty($comment['profile_image'])): ?>
                                                <img src="<?php echo htmlspecialchars($comment['profile_image']); ?>" 
                                                     alt="<?php echo htmlspecialchars($comment['user_name']); ?>" 
                                                     class="rounded-circle" width="50" height="50">
                                            <?php else: ?>
                                                <div class="avatar-placeholder rounded-circle bg-secondary text-white d-flex align-items-center justify-content-center" 
                                                     style="width: 50px; height: 50px;">
                                                    <?php echo strtoupper(substr($comment['user_name'], 0, 1)); ?>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="comment-content flex-grow-1">
                                            <div class="d-flex justify-content-between">
                                                <h5 class="mb-1"><?php echo htmlspecialchars($comment['user_name']); ?></h5>
                                                <small class="text-muted">
                                                    <?php echo date('M j, Y g:i A', strtotime($comment['created_at'])); ?>
                                                </small>
                                            </div>
                                            <p class="mb-1"><?php echo nl2br(htmlspecialchars($comment['comment'])); ?></p>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="col-lg-4">
                <!-- Related Content -->
                <div class="related-content">
                    <h3 class="mb-4"><?php echo $lang_data['related_content']; ?></h3>
                    
                    <?php if (empty($relatedContent)): ?>
                        <div class="text-center text-muted my-5">
                            <p><?php echo $lang_data['no_related']; ?></p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($relatedContent as $related): ?>
                            <div class="card mb-3">
                                <div class="row g-0">
                                    <div class="col-md-4">
                                        <img src="<?php echo htmlspecialchars($related['thumbnail']); ?>" 
                                             class="img-fluid rounded-start" 
                                             alt="<?php echo htmlspecialchars($related['title']); ?>">
                                    </div>
                                    <div class="col-md-8">
                                        <div class="card-body">
                                            <h5 class="card-title"><?php echo htmlspecialchars($related['title']); ?></h5>
                                            <p class="card-text small text-muted">
                                                <i class="far fa-calendar-alt me-1"></i> <?php echo date('M j, Y', strtotime($related['created_at'])); ?>
                                            </p>
                                            <a href="watch.php?id=<?php echo $related['id']; ?>" class="stretched-link"></a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                
                <!-- Categories -->
                <div class="categories mt-5">
                    <h3 class="mb-4"><?php echo $lang_data['categories']; ?></h3>
                    <div class="list-group">
                        <?php
                        // Get categories
                        $categories = $contentManager->getCategories();
                        
                        foreach ($categories as $category):
                        ?>
                            <a href="content.php?category=<?php echo $category['id']; ?>" 
                               class="list-group-item list-group-item-action d-flex justify-content-between align-items-center">
                                <?php echo htmlspecialchars($category['name']); ?>
                                <span class="badge bg-primary rounded-pill"><?php echo $category['content_count']; ?></span>
                            </a>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<script>
// Function to share content on social media
function shareContent(platform) {
    const url = encodeURIComponent(window.location.href);
    const title = encodeURIComponent('<?php echo htmlspecialchars($contentItem['title']); ?>');
    
    let shareUrl = '';
    
    switch (platform) {
        case 'facebook':
            shareUrl = `https://www.facebook.com/sharer/sharer.php?u=${url}`;
            break;
        case 'twitter':
            shareUrl = `https://twitter.com/intent/tweet?url=${url}&text=${title}`;
            break;
        case 'whatsapp':
            shareUrl = `https://api.whatsapp.com/send?text=${title} ${url}`;
            break;
    }
    
    window.open(shareUrl, '_blank', 'width=600,height=400');
    return false;
}
</script>

<?php
// Include footer
include 'includes/footer.php';
?>
