<?php
/**
 * Shows Page
 * Displays all available shows and content for users to browse and purchase
 */

// Include configuration
require_once 'includes/config.php';

// Include required classes
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/content.php';

// Initialize classes
$auth = new Auth();
$contentManager = new Content();
$db = Database::getInstance();

// Check if user is logged in
$isLoggedIn = $auth->isLoggedIn();
$currentUser = $isLoggedIn ? $auth->getCurrentUser() : null;

// Get filter parameters
$category = isset($_GET['category']) ? (int)$_GET['category'] : 0;
$type = isset($_GET['type']) ? $_GET['type'] : '';
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 12;

// Get content categories
$categories = $contentManager->getCategories();

// Get content based on filters
if (!empty($search)) {
    // Search for content
    $content = $contentManager->searchContent($search, $perPage, ($page - 1) * $perPage);
    $totalContent = $db->getValue(
        "SELECT COUNT(*) FROM content WHERE status = 'published' AND (title LIKE ? OR description LIKE ? OR tags LIKE ?)",
        ["%{$search}%", "%{$search}%", "%{$search}%"]
    );
} elseif ($category > 0) {
    // Filter by category
    $content = $contentManager->getContentByCategory($category, $perPage, ($page - 1) * $perPage);
    $totalContent = $db->getValue(
        "SELECT COUNT(*) FROM content WHERE status = 'published' AND category_id = ?",
        [$category]
    );
} elseif (!empty($type)) {
    // Filter by content type
    $content = $contentManager->getAllContent($type, 'published', $perPage, ($page - 1) * $perPage);
    $totalContent = $db->getValue(
        "SELECT COUNT(*) FROM content WHERE status = 'published' AND content_type = ?",
        [$type]
    );
} else {
    // Get all content
    $content = $contentManager->getAllContent(null, 'published', $perPage, ($page - 1) * $perPage);
    $totalContent = $db->getValue("SELECT COUNT(*) FROM content WHERE status = 'published'");
}

// Get featured content for showcase
$featuredContent = $contentManager->getFeaturedContent(3);

// Calculate total pages
$totalPages = ceil($totalContent / $perPage);

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "languages/{$lang}.php";

// Set page title
$pageTitle = $lang_data['shows'] . ' - ' . SITE_NAME;

// Include header
include 'includes/header.php';
?>

<!-- Shows Page -->
<section class="shows-hero">
    <div class="container">
        <div class="row">
            <div class="col-lg-12 text-center">
                <h1 class="display-4 fw-bold"><?php echo $lang_data['shows']; ?></h1>
                <p class="lead"><?php echo isset($lang_data['shows_subtitle']) ? $lang_data['shows_subtitle'] : 'Browse our collection of original programming and client productions'; ?></p>
                
                <!-- Search Form -->
                <div class="search-container mx-auto mt-4">
                    <form action="shows.php" method="get" class="search-form">
                        <div class="input-group">
                            <input type="text" class="form-control" name="search" placeholder="<?php echo isset($lang_data['search_shows']) ? $lang_data['search_shows'] : 'Search shows...'; ?>" value="<?php echo htmlspecialchars($search); ?>">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-search me-2"></i><?php echo $lang_data['search']; ?>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Featured Content Carousel - Only show if no search/filter is active -->
<?php if (empty($search) && $category == 0 && empty($type) && !empty($featuredContent)): ?>
<section class="featured-content">
    <div class="container">
        <div class="row">
            <div class="col-12">
                <div id="featuredCarousel" class="carousel slide" data-bs-ride="carousel">
                    <div class="carousel-inner">
                        <?php foreach ($featuredContent as $index => $item): ?>
                            <div class="carousel-item <?php echo ($index === 0) ? 'active' : ''; ?>">
                                <div class="featured-item">
                                    <div class="row align-items-center">
                                        <div class="col-lg-6">
                                            <img src="<?php echo htmlspecialchars($item['thumbnail']); ?>" 
                                                 class="img-fluid rounded featured-image" 
                                                 alt="<?php echo htmlspecialchars($item['title']); ?>">
                                        </div>
                                        <div class="col-lg-6">
                                            <div class="featured-content-text p-4">
                                                <?php if (isset($item['content_type'])): ?>
                                                    <div class="content-type mb-2">
                                                        <span class="badge bg-primary">
                                                            <?php echo ucfirst(htmlspecialchars($item['content_type'])); ?>
                                                        </span>
                                                    </div>
                                                <?php endif; ?>
                                                
                                                <h2 class="featured-title"><?php echo htmlspecialchars($item['title']); ?></h2>
                                                
                                                <p class="featured-description">
                                                    <?php echo nl2br(htmlspecialchars(substr($item['description'], 0, 250) . (strlen($item['description']) > 250 ? '...' : ''))); ?>
                                                </p>
                                                
                                                <div class="featured-meta mt-3 mb-4">
                                                    <span class="me-3">
                                                        <i class="fas fa-eye me-1"></i> 
                                                        <?php echo number_format($item['view_count']); ?> <?php echo $lang_data['views']; ?>
                                                    </span>
                                                    
                                                    <?php if (isset($item['duration'])): ?>
                                                        <span>
                                                            <i class="fas fa-clock me-1"></i> 
                                                            <?php echo $item['duration']; ?> <?php echo isset($lang_data['minutes']) ? $lang_data['minutes'] : 'minutes'; ?>
                                                        </span>
                                                    <?php endif; ?>
                                                </div>
                                                
                                                <a href="watch.php?id=<?php echo $item['id']; ?>" class="btn btn-primary btn-lg">
                                                    <i class="fas fa-play me-2"></i><?php echo $lang_data['watch_now']; ?>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <button class="carousel-control-prev" type="button" data-bs-target="#featuredCarousel" data-bs-slide="prev">
                        <span class="carousel-control-prev-icon" aria-hidden="true"></span>
                        <span class="visually-hidden"><?php echo $lang_data['previous']; ?></span>
                    </button>
                    <button class="carousel-control-next" type="button" data-bs-target="#featuredCarousel" data-bs-slide="next">
                        <span class="carousel-control-next-icon" aria-hidden="true"></span>
                        <span class="visually-hidden"><?php echo $lang_data['next']; ?></span>
                    </button>
                    
                    <div class="carousel-indicators">
                        <?php foreach ($featuredContent as $index => $item): ?>
                            <button type="button" data-bs-target="#featuredCarousel" data-bs-slide-to="<?php echo $index; ?>" 
                                    <?php echo ($index === 0) ? 'class="active" aria-current="true"' : ''; ?> 
                                    aria-label="Slide <?php echo $index + 1; ?>"></button>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>
<?php endif; ?>

<section class="shows-content py-5">
    <div class="container">
        <div class="row">
            <!-- Sidebar Filters -->
            <div class="col-lg-3">
                <div class="filters-sidebar">
                    <div class="card border-0 shadow-sm mb-4">
                        <div class="card-header bg-white py-3">
                            <h5 class="mb-0"><?php echo isset($lang_data['categories']) ? $lang_data['categories'] : 'Categories'; ?></h5>
                        </div>
                        <div class="card-body p-0">
                            <div class="list-group list-group-flush rounded-0">
                                <a href="shows.php" class="list-group-item list-group-item-action d-flex justify-content-between align-items-center <?php echo ($category == 0 && empty($type)) ? 'active' : ''; ?>">
                                    <?php echo isset($lang_data['all_categories']) ? $lang_data['all_categories'] : 'All Categories'; ?>
                                    <span class="badge bg-primary rounded-pill"><?php echo $totalContent; ?></span>
                                </a>
                                
                                <?php foreach ($categories as $cat): ?>
                                    <a href="shows.php?category=<?php echo $cat['id']; ?>" 
                                       class="list-group-item list-group-item-action d-flex justify-content-between align-items-center <?php echo ($category == $cat['id']) ? 'active' : ''; ?>">
                                        <?php echo htmlspecialchars($cat['name']); ?>
                                        <span class="badge bg-primary rounded-pill"><?php echo isset($cat['content_count']) ? $cat['content_count'] : 0; ?></span>
                                    </a>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card border-0 shadow-sm mb-4">
                        <div class="card-header bg-white py-3">
                            <h5 class="mb-0"><?php echo isset($lang_data['content_type']) ? $lang_data['content_type'] : 'Content Type'; ?></h5>
                        </div>
                        <div class="card-body p-0">
                            <div class="list-group list-group-flush rounded-0">
                                <a href="shows.php?type=show" class="list-group-item list-group-item-action <?php echo ($type == 'show') ? 'active' : ''; ?>">
                                    <i class="fas fa-tv me-2"></i> <?php echo isset($lang_data['tv_shows']) ? $lang_data['tv_shows'] : 'TV Shows'; ?>
                                </a>
                                <a href="shows.php?type=interview" class="list-group-item list-group-item-action <?php echo ($type == 'interview') ? 'active' : ''; ?>">
                                    <i class="fas fa-microphone me-2"></i> <?php echo isset($lang_data['interviews']) ? $lang_data['interviews'] : 'Interviews'; ?>
                                </a>
                                <a href="shows.php?type=music" class="list-group-item list-group-item-action <?php echo ($type == 'music') ? 'active' : ''; ?>">
                                    <i class="fas fa-music me-2"></i> <?php echo isset($lang_data['music_videos']) ? $lang_data['music_videos'] : 'Music Videos'; ?>
                                </a>
                                <a href="shows.php?type=documentary" class="list-group-item list-group-item-action <?php echo ($type == 'documentary') ? 'active' : ''; ?>">
                                    <i class="fas fa-film me-2"></i> <?php echo isset($lang_data['documentaries']) ? $lang_data['documentaries'] : 'Documentaries'; ?>
                                </a>
                                <a href="shows.php?type=podcast" class="list-group-item list-group-item-action <?php echo ($type == 'podcast') ? 'active' : ''; ?>">
                                    <i class="fas fa-podcast me-2"></i> <?php echo isset($lang_data['podcasts']) ? $lang_data['podcasts'] : 'Podcasts'; ?>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Content Grid -->
            <div class="col-lg-9">
                <!-- Filter Results Header -->
                <div class="content-header mb-4">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <?php if (!empty($search)): ?>
                                <h4><?php echo isset($lang_data['search_results']) ? $lang_data['search_results'] : 'Search Results'; ?>: "<?php echo htmlspecialchars($search); ?>"</h4>
                            <?php elseif ($category > 0): ?>
                                <?php 
                                $categoryName = '';
                                foreach ($categories as $cat) {
                                    if ($cat['id'] == $category) {
                                        $categoryName = $cat['name'];
                                        break;
                                    }
                                }
                                ?>
                                <h4><?php echo htmlspecialchars($categoryName); ?></h4>
                            <?php elseif (!empty($type)): ?>
                                <h4>
                                    <?php 
                                    switch ($type) {
                                        case 'show':
                                            echo isset($lang_data['tv_shows']) ? $lang_data['tv_shows'] : 'TV Shows';
                                            break;
                                        case 'interview':
                                            echo isset($lang_data['interviews']) ? $lang_data['interviews'] : 'Interviews';
                                            break;
                                        case 'music':
                                            echo isset($lang_data['music_videos']) ? $lang_data['music_videos'] : 'Music Videos';
                                            break;
                                        case 'documentary':
                                            echo isset($lang_data['documentaries']) ? $lang_data['documentaries'] : 'Documentaries';
                                            break;
                                        case 'podcast':
                                            echo isset($lang_data['podcasts']) ? $lang_data['podcasts'] : 'Podcasts';
                                            break;
                                        default:
                                            echo ucfirst($type);
                                    }
                                    ?>
                                </h4>
                            <?php else: ?>
                                <h4><?php echo isset($lang_data['all_shows']) ? $lang_data['all_shows'] : 'All Shows'; ?></h4>
                            <?php endif; ?>
                            
                            <p class="text-muted"><?php echo sprintf(isset($lang_data['showing_results']) ? $lang_data['showing_results'] : 'Showing %d of %d results', count($content), $totalContent); ?></p>
                        </div>
                        
                        <div class="d-none d-md-block">
                            <div class="btn-group" role="group" aria-label="View options">
                                <button type="button" class="btn btn-outline-primary active" id="grid-view-btn">
                                    <i class="fas fa-th-large"></i>
                                </button>
                                <button type="button" class="btn btn-outline-primary" id="list-view-btn">
                                    <i class="fas fa-list"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php if (empty($content)): ?>
                    <!-- No Results -->
                    <div class="no-results text-center py-5">
                        <img src="assets/images/no-results.svg" alt="No Results" class="img-fluid mb-3" style="max-height: 150px;">
                        <h3><?php echo isset($lang_data['no_shows_found']) ? $lang_data['no_shows_found'] : 'No Shows Found'; ?></h3>
                        <p class="text-muted"><?php echo isset($lang_data['no_shows_message']) ? $lang_data['no_shows_message'] : 'We couldn\'t find any shows matching your search criteria.'; ?></p>
                        <a href="shows.php" class="btn btn-primary mt-3">
                            <i class="fas fa-sync me-2"></i><?php echo isset($lang_data['clear_filters']) ? $lang_data['clear_filters'] : 'Clear Filters'; ?>
                        </a>
                    </div>
                <?php else: ?>
                    <!-- Content Grid View -->
                    <div class="content-grid" id="grid-view">
                        <div class="row row-cols-1 row-cols-md-2 row-cols-xl-3 g-4">
                            <?php foreach ($content as $item): ?>
                                <div class="col">
                                    <div class="card h-100 content-card">
                                        <div class="content-thumbnail ratio ratio-16x9">
                                            <img src="<?php echo htmlspecialchars($item['thumbnail']); ?>" 
                                                 class="card-img-top object-fit-cover" 
                                                 alt="<?php echo htmlspecialchars($item['title']); ?>">
                                            <div class="content-overlay">
                                                <a href="watch.php?id=<?php echo $item['id']; ?>" class="btn btn-play">
                                                    <i class="fas fa-play"></i>
                                                </a>
                                            </div>
                                            <?php if (isset($item['price']) && $item['price'] > 0): ?>
                                                <div class="content-badge">
                                                    <span class="badge bg-primary">
                                                        <?php echo number_format($item['price']); ?> <?php echo CURRENCY; ?>
                                                    </span>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="card-body">
                                            <h5 class="card-title"><?php echo htmlspecialchars($item['title']); ?></h5>
                                            <p class="card-text small text-truncate-3">
                                                <?php echo htmlspecialchars(substr($item['description'], 0, 120) . '...'); ?>
                                            </p>
                                        </div>
                                        <div class="card-footer bg-white border-top-0">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <small class="text-muted">
                                                    <i class="fas fa-eye me-1"></i> <?php echo number_format($item['view_count']); ?>
                                                </small>
                                                <a href="watch.php?id=<?php echo $item['id']; ?>" class="btn btn-sm btn-primary">
                                                    <?php echo $lang_data['watch_now']; ?>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Content List View (Hidden by Default) -->
                    <div class="content-list d-none" id="list-view">
                        <?php foreach ($content as $item): ?>
                            <div class="card mb-3 content-list-item">
                                <div class="row g-0">
                                    <div class="col-md-4">
                                        <div class="content-thumbnail ratio ratio-16x9 h-100">
                                            <img src="<?php echo htmlspecialchars($item['thumbnail']); ?>" 
                                                 class="img-fluid rounded-start object-fit-cover" 
                                                 alt="<?php echo htmlspecialchars($item['title']); ?>">
                                            <div class="content-overlay">
                                                <a href="watch.php?id=<?php echo $item['id']; ?>" class="btn btn-play">
                                                    <i class="fas fa-play"></i>
                                                </a>
                                            </div>
                                            <?php if (isset($item['price']) && $item['price'] > 0): ?>
                                                <div class="content-badge">
                                                    <span class="badge bg-primary">
                                                        <?php echo number_format($item['price']); ?> <?php echo CURRENCY; ?>
                                                    </span>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="col-md-8">
                                        <div class="card-body">
                                            <h5 class="card-title"><?php echo htmlspecialchars($item['title']); ?></h5>
                                            <p class="card-text">
                                                <?php echo htmlspecialchars(substr($item['description'], 0, 200) . '...'); ?>
                                            </p>
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div class="text-muted small">
                                                    <span class="me-3">
                                                        <i class="fas fa-eye me-1"></i> <?php echo number_format($item['view_count']); ?> <?php echo $lang_data['views']; ?>
                                                    </span>
                                                    <?php if (isset($item['author_name'])): ?>
                                                        <span>
                                                            <i class="fas fa-user me-1"></i> <?php echo htmlspecialchars($item['author_name']); ?>
                                                        </span>
                                                    <?php endif; ?>
                                                </div>
                                                <a href="watch.php?id=<?php echo $item['id']; ?>" class="btn btn-primary">
                                                    <?php echo $lang_data['watch_now']; ?>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <nav aria-label="Page navigation" class="mt-4">
                            <ul class="pagination justify-content-center">
                                <!-- Previous Page -->
                                <li class="page-item <?php echo ($page <= 1) ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="<?php echo ($page <= 1) ? '#' : 'shows.php?' . http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>" aria-label="Previous">
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>
                                </li>
                                
                                <!-- Page Numbers -->
                                <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                                    <li class="page-item <?php echo ($page == $i) ? 'active' : ''; ?>">
                                        <a class="page-link" href="<?php echo 'shows.php?' . http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                                            <?php echo $i; ?>
                                        </a>
                                    </li>
                                <?php endfor; ?>
                                
                                <!-- Next Page -->
                                <li class="page-item <?php echo ($page >= $totalPages) ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="<?php echo ($page >= $totalPages) ? '#' : 'shows.php?' . http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>" aria-label="Next">
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </li>
                            </ul>
                        </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</section>

<!-- Shows Page Styles -->
<style>
    /* Hero Section */
    .shows-hero {
        background-color: var(--bs-primary);
        color: white;
        padding: 60px 0;
        position: relative;
    }
    
    .shows-hero::after {
        content: '';
        position: absolute;
        bottom: 0;
        left: 0;
        right: 0;
        height: 50px;
        background-image: url('data:image/svg+xml;charset=utf8,%3Csvg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1440 48"%3E%3Cpath fill="%23ffffff" fill-opacity="1" d="M0,32L120,32C240,32,480,32,720,16C960,0,1200,0,1320,0L1440,0L1440,48L1320,48C1200,48,960,48,720,48C480,48,240,48,120,48L0,48Z"%3E%3C/path%3E%3C/svg%3E');
        background-size: cover;
        background-position: center;
    }
    
    .search-container {
        max-width: 600px;
    }
    
    /* Featured Content */
    .featured-content {
        padding: 80px 0;
        background-color: #f8f9fa;
    }
    
    .featured-item {
        background-color: white;
        border-radius: 10px;
        overflow: hidden;
        box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    }
    
    .featured-image {
        height: 100%;
        object-fit: cover;
    }
    
    .featured-title {
        font-size: 2.5rem;
        font-weight: bold;
        margin-bottom: 1rem;
    }
    
    .carousel-indicators {
        bottom: -50px;
    }
    
    .carousel-indicators button {
        width: 10px;
        height: 10px;
        border-radius: 50%;
        margin: 0 5px;
    }
    
    .carousel-control-prev, .carousel-control-next {
        width: 5%;
        opacity: 0.8;
    }
    
    /* Content Cards */
    .content-card {
        transition: all 0.3s ease;
        border: none;
        overflow: hidden;
    }
    
    .content-card:hover {
        box-shadow: 0 .5rem 1rem rgba(0,0,0,.15)!important;
        transform: translateY(-5px);
    }
    
    .content-thumbnail {
        position: relative;
        overflow: hidden;
    }
    
    .content-overlay {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.3);
        display: flex;
        align-items: center;
        justify-content: center;
        opacity: 0;
        transition: opacity 0.3s ease;
    }
    
    .content-thumbnail:hover .content-overlay {
        opacity: 1;
    }
    
    .btn-play {
        width: 50px;
        height: 50px;
        border-radius: 50%;
        background-color: rgba(255, 255, 255, 0.9);
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--bs-primary);
        font-size: 1.2rem;
        transition: all 0.3s ease;
    }
    
    .btn-play:hover {
        background-color: var(--bs-primary);
        color: white;
        transform: scale(1.1);
    }
    
    .content-badge {
        position: absolute;
        top: 10px;
        right: 10px;
    }
    
    /* List View */
    .content-list-item {
        transition: all 0.3s ease;
        border: none;
        overflow: hidden;
    }
    
    .content-list-item:hover {
        box-shadow: 0 .5rem 1rem rgba(0,0,0,.15)!important;
    }
    
    /* Text truncation for multi-line */
    .text-truncate-3 {
        display: -webkit-box;
        -webkit-line-clamp: 3;
        -webkit-box-orient: vertical;
        overflow: hidden;
        text-overflow: ellipsis;
    }
</style>

<!-- Initialize Shows Page JavaScript -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Grid/List view toggle
    const gridViewBtn = document.getElementById('grid-view-btn');
    const listViewBtn = document.getElementById('list-view-btn');
    const gridView = document.getElementById('grid-view');
    const listView = document.getElementById('list-view');
    
    if (gridViewBtn && listViewBtn && gridView && listView) {
        gridViewBtn.addEventListener('click', function() {
            gridView.classList.remove('d-none');
            listView.classList.add('d-none');
            gridViewBtn.classList.add('active');
            listViewBtn.classList.remove('active');
        });
        
        listViewBtn.addEventListener('click', function() {
            gridView.classList.add('d-none');
            listView.classList.remove('d-none');
            gridViewBtn.classList.remove('active');
            listViewBtn.classList.add('active');
        });
    }
});
</script>

<?php
// Include footer
include 'includes/footer.php';
?>