<?php
/**
 * User Registration Page - 100% Error Protected with Session Fix
 */

// For debugging (can be removed in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// REMOVED session_start() - Let config.php handle session initialization

// Try to include configuration
if (file_exists('includes/config.php')) {
    require_once 'includes/config.php';
} else {
    die("Configuration file not found.");
}

// Try to include required classes
if (file_exists('includes/db.php') && file_exists('includes/auth.php')) {
    require_once 'includes/db.php';
    require_once 'includes/auth.php';
} else {
    die("Required class files not found.");
}

// Initialize classes
try {
    $auth = new Auth();
} catch (Exception $e) {
    die("Failed to initialize authentication: " . $e->getMessage());
}

// Check if user is already logged in
if ($auth->isLoggedIn()) {
    // Redirect to home page or requested page
    $redirect = isset($_GET['redirect']) ? $_GET['redirect'] : 'index.php';
    header('Location: ' . $redirect);
    exit;
}

// Process registration form
$registerError = '';
$registerSuccess = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // SAFELY get POST variables with isset checks
    $name = isset($_POST['name']) ? trim($_POST['name']) : '';
    $email = isset($_POST['email']) ? trim($_POST['email']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $password = isset($_POST['password']) ? $_POST['password'] : '';  // No trim() for password
    $confirmPassword = isset($_POST['confirm_password']) ? $_POST['confirm_password'] : '';
    $agreeTerms = isset($_POST['agree_terms']) && $_POST['agree_terms'] == '1';
    $csrf_token = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';

    // Verify CSRF token
    if (empty($csrf_token) || !$auth->verifyCsrfToken($csrf_token)) {
        $registerError = 'Invalid security token. Please try again.';
    } else {
        // Validate inputs
        if (empty($name) || empty($email) || empty($password) || empty($confirmPassword)) {
            $registerError = 'All fields are required.';
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $registerError = 'Please enter a valid email address.';
        } elseif ($password !== $confirmPassword) {
            $registerError = 'Passwords do not match.';
        } elseif (strlen($password) < 8) {
            $registerError = 'Password must be at least 8 characters long.';
        } elseif (!$agreeTerms) {
            $registerError = 'You must agree to the Terms and Conditions.';
        } else {
            try {
                // Prepare user data - ensure all fields are set
                $userData = [
                    'name' => $name,
                    'email' => $email,
                    'phone' => $phone,
                    'password' => $password,  // Will be hashed in the register method
                    'role' => 'customer',
                    'status' => 'active'
                ];
                
                // Attempt registration
                $result = $auth->register($userData);
                
                if (isset($result['success']) && $result['success']) {
                    $registerSuccess = true;
                    
                    // Redirect after successful registration
                    $redirect = isset($_GET['redirect']) ? $_GET['redirect'] : 'index.php';
                    
                    // Set a flash message
                    $_SESSION['flash_message'] = 'Registration successful! Welcome to ' . (defined('SITE_NAME') ? SITE_NAME : 'Web TV Studio');
                    $_SESSION['flash_type'] = 'success';
                    
                    header('Location: ' . $redirect);
                    exit;
                } else {
                    $registerError = isset($result['message']) ? $result['message'] : 'Registration failed';
                }
            } catch (Exception $e) {
                $registerError = 'An error occurred during registration: ' . $e->getMessage();
                error_log("Registration error: " . $e->getMessage());
            }
        }
    }
}

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : (defined('DEFAULT_LANGUAGE') ? DEFAULT_LANGUAGE : 'en');
$langFilePath = "languages/{$lang}.php";

if (file_exists($langFilePath)) {
    require_once $langFilePath;
} else {
    // Fallback language data
    $lang_data = [
        'register_title' => 'Create Account',
        'register_message' => 'Create an account to book our studio and access premium content.',
        'full_name' => 'Full Name',
        'email_address' => 'Email Address',
        'phone_number' => 'Phone Number',
        'password' => 'Password',
        'confirm_password' => 'Confirm Password',
        'agree_terms' => 'I agree to the Terms and Conditions',
        'register' => 'Register',
        'already_have_account' => 'Already have an account?',
        'login' => 'Login'
    ];
}

// Set page title
$pageTitle = (isset($lang_data['register_title']) ? $lang_data['register_title'] : 'Create Account') . ' - ' . (defined('SITE_NAME') ? SITE_NAME : 'Web TV Studio');

// Try to include header
$headerIncluded = @include 'includes/header.php';

// If header failed, create a basic one
if (!$headerIncluded) {
    echo '<!DOCTYPE html>';
    echo '<html lang="' . $lang . '">';
    echo '<head>';
    echo '<meta charset="UTF-8">';
    echo '<meta name="viewport" content="width=device-width, initial-scale=1.0">';
    echo '<title>' . $pageTitle . '</title>';
    echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">';
    echo '</head>';
    echo '<body>';
}
?>

<section class="register-section py-5">
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-body p-4">
                        <div class="text-center mb-4">
                            <h2><?php echo isset($lang_data['register_title']) ? $lang_data['register_title'] : 'Create Account'; ?></h2>
                            <p class="text-muted"><?php echo isset($lang_data['register_message']) ? $lang_data['register_message'] : 'Create an account to book our studio and access premium content.'; ?></p>
                        </div>
                        
                        <?php if (!empty($registerError)): ?>
                            <div class="alert alert-danger">
                                <?php echo $registerError; ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($registerSuccess): ?>
                            <div class="alert alert-success">
                                Registration successful! Please <a href="login.php">login</a> to continue.
                            </div>
                        <?php else: ?>
                            <form action="register.php<?php echo isset($_GET['redirect']) ? '?redirect=' . urlencode($_GET['redirect']) : ''; ?>" method="post">
                                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                                
                                <div class="mb-3">
                                    <label for="name" class="form-label"><?php echo isset($lang_data['full_name']) ? $lang_data['full_name'] : 'Full Name'; ?></label>
                                    <input type="text" class="form-control" id="name" name="name" required 
                                           value="<?php echo isset($_POST['name']) ? htmlspecialchars($_POST['name']) : ''; ?>">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="email" class="form-label"><?php echo isset($lang_data['email_address']) ? $lang_data['email_address'] : 'Email Address'; ?></label>
                                    <input type="email" class="form-control" id="email" name="email" required 
                                           value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="phone" class="form-label"><?php echo isset($lang_data['phone_number']) ? $lang_data['phone_number'] : 'Phone Number'; ?></label>
                                    <input type="tel" class="form-control" id="phone" name="phone" 
                                           value="<?php echo isset($_POST['phone']) ? htmlspecialchars($_POST['phone']) : ''; ?>">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="password" class="form-label"><?php echo isset($lang_data['password']) ? $lang_data['password'] : 'Password'; ?></label>
                                    <input type="password" class="form-control" id="password" name="password" required>
                                    <small class="form-text text-muted">Password must be at least 8 characters long.</small>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="confirm_password" class="form-label"><?php echo isset($lang_data['confirm_password']) ? $lang_data['confirm_password'] : 'Confirm Password'; ?></label>
                                    <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                                </div>
                                
                                <div class="mb-3 form-check">
                                    <input type="checkbox" class="form-check-input" id="agree_terms" name="agree_terms" value="1" required
                                           <?php echo isset($_POST['agree_terms']) ? 'checked' : ''; ?>>
                                    <label class="form-check-label" for="agree_terms"><?php echo isset($lang_data['agree_terms']) ? $lang_data['agree_terms'] : 'I agree to the Terms and Conditions'; ?></label>
                                </div>
                                
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-primary"><?php echo isset($lang_data['register']) ? $lang_data['register'] : 'Register'; ?></button>
                                </div>
                            </form>
                        <?php endif; ?>
                        
                        <hr class="my-4">
                        
                        <div class="text-center">
                            <p><?php echo isset($lang_data['already_have_account']) ? $lang_data['already_have_account'] : 'Already have an account?'; ?></p>
                            <a href="login.php<?php echo isset($_GET['redirect']) ? '?redirect=' . urlencode($_GET['redirect']) : ''; ?>" 
                               class="btn btn-outline-primary"><?php echo isset($lang_data['login']) ? $lang_data['login'] : 'Login'; ?></a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<?php
// Try to include footer
$footerIncluded = @include 'includes/footer.php';

// If footer failed, create a basic one
if (!$footerIncluded) {
    echo '<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>';
    echo '</body>';
    echo '</html>';
}
?>