<?php
/**
 * User Profile Page
 * Allows users to view and edit their profile information
 */

// Include configuration
require_once 'includes/config.php';

// Include required classes
require_once 'includes/db.php';
require_once 'includes/auth.php';

// Initialize auth class
$auth = new Auth();

// Check if user is logged in
if (!$auth->isLoggedIn()) {
    // Redirect to login page
    header('Location: login.php?redirect=' . urlencode($_SERVER['PHP_SELF']));
    exit;
}

// Get current user
$currentUser = $auth->getCurrentUser();

// Process profile update
$updateSuccess = false;
$updateError = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
    // Initialize form variables
    $name = isset($_POST['name']) ? trim($_POST['name']) : '';
    $email = isset($_POST['email']) ? trim($_POST['email']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $csrf_token = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';
    
    // Verify CSRF token
    if (empty($csrf_token) || !$auth->verifyCsrfToken($csrf_token)) {
        $updateError = 'Invalid security token. Please try again.';
    } 
    // Validate inputs
    elseif (empty($name)) {
        $updateError = 'Name is required.';
    } 
    elseif (empty($email)) {
        $updateError = 'Email is required.';
    } 
    elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $updateError = 'Please enter a valid email address.';
    } 
    else {
        // Prepare user data
        $userData = [
            'name' => $name,
            'email' => $email,
            'phone' => $phone
        ];
        
        // Update user
        $result = $auth->updateUser($currentUser['id'], $userData);
        
        if ($result) {
            $updateSuccess = true;
            $currentUser = $auth->getCurrentUser(); // Refresh user data
        } else {
            $updateError = 'Failed to update profile. Please try again.';
        }
    }
}

// Process password change
$passwordSuccess = false;
$passwordError = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    // Initialize form variables
    $currentPassword = isset($_POST['current_password']) ? $_POST['current_password'] : '';
    $newPassword = isset($_POST['new_password']) ? $_POST['new_password'] : '';
    $confirmPassword = isset($_POST['confirm_password']) ? $_POST['confirm_password'] : '';
    $csrf_token = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';
    
    // Verify CSRF token
    if (empty($csrf_token) || !$auth->verifyCsrfToken($csrf_token)) {
        $passwordError = 'Invalid security token. Please try again.';
    } 
    // Validate inputs
    elseif (empty($currentPassword)) {
        $passwordError = 'Current password is required.';
    } 
    elseif (empty($newPassword)) {
        $passwordError = 'New password is required.';
    } 
    elseif (strlen($newPassword) < 8) {
        $passwordError = 'New password must be at least 8 characters long.';
    } 
    elseif ($newPassword !== $confirmPassword) {
        $passwordError = 'New passwords do not match.';
    } 
    else {
        // Verify current password
        $db = Database::getInstance();
        $user = $db->getRow("SELECT * FROM users WHERE id = ?", [$currentUser['id']]);
        
        if (!$user || !password_verify($currentPassword, $user['password'])) {
            $passwordError = 'Current password is incorrect.';
        } else {
            // Update password
            $result = $auth->updateUser($currentUser['id'], [
                'password' => $newPassword
            ]);
            
            if ($result) {
                $passwordSuccess = true;
            } else {
                $passwordError = 'Failed to change password. Please try again.';
            }
        }
    }
}

// Get user bookings
try {
    $db = Database::getInstance();
    
    // Check if bookings table exists
    $bookingsTableExists = false;
    try {
        $tableCheck = $db->getRow("SHOW TABLES LIKE 'bookings'");
        $bookingsTableExists = !empty($tableCheck);
    } catch (Exception $e) {
        // Table doesn't exist
        error_log("Bookings table check failed: " . $e->getMessage());
    }
    
    if ($bookingsTableExists) {
        $bookings = $db->getRows(
            "SELECT b.*, s.name as service_name 
             FROM bookings b 
             LEFT JOIN services s ON b.service_id = s.id
             WHERE b.user_id = ? 
             ORDER BY b.booking_date DESC", 
            [$currentUser['id']]
        ) ?: [];
    } else {
        $bookings = [];
    }
} catch (Exception $e) {
    error_log("Error getting user bookings: " . $e->getMessage());
    $bookings = [];
}

// Get user purchased content
try {
    // Check if content_payments table exists
    $contentPaymentsTableExists = false;
    try {
        $tableCheck = $db->getRow("SHOW TABLES LIKE 'content_payments'");
        $contentPaymentsTableExists = !empty($tableCheck);
    } catch (Exception $e) {
        // Table doesn't exist
        error_log("Content payments table check failed: " . $e->getMessage());
    }
    
    if ($contentPaymentsTableExists) {
        $purchases = $db->getRows(
            "SELECT cp.*, c.title, c.thumbnail, c.description 
             FROM content_payments cp 
             LEFT JOIN content c ON cp.content_id = c.id
             WHERE cp.user_id = ? AND cp.status = 'completed'
             ORDER BY cp.created_at DESC", 
            [$currentUser['id']]
        ) ?: [];
    } else {
        $purchases = [];
    }
} catch (Exception $e) {
    error_log("Error getting user purchased content: " . $e->getMessage());
    $purchases = [];
}

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "languages/{$lang}.php";

// Make sure all required language variables are set
if (!isset($lang_data['account_status'])) {
    $lang_data['account_status'] = 'Account Status';
}
if (!isset($lang_data['active'])) {
    $lang_data['active'] = 'Active';
}
if (!isset($lang_data['member_since'])) {
    $lang_data['member_since'] = 'Member Since';
}
if (!isset($lang_data['purchased_on'])) {
    $lang_data['purchased_on'] = 'Purchased on';
}
if (!isset($lang_data['statistics'])) {
    $lang_data['statistics'] = 'Statistics';
}
if (!isset($lang_data['bookings'])) {
    $lang_data['bookings'] = 'Bookings';
}
if (!isset($lang_data['purchased_content'])) {
    $lang_data['purchased_content'] = 'Purchased Content';
}
if (!isset($lang_data['view_my_bookings'])) {
    $lang_data['view_my_bookings'] = 'View My Bookings';
}
if (!isset($lang_data['book_now'])) {
    $lang_data['book_now'] = 'Book Now';
}
if (!isset($lang_data['view_details'])) {
    $lang_data['view_details'] = 'View Details';
}
if (!isset($lang_data['cancel_booking'])) {
    $lang_data['cancel_booking'] = 'Cancel Booking';
}
if (!isset($lang_data['browse_shows'])) {
    $lang_data['browse_shows'] = 'Browse Shows';
}

// Set page title
$pageTitle = $lang_data['my_profile'] . ' - ' . SITE_NAME;

// Include header
include 'includes/header.php';
?>

<!-- User Profile Page - Redesigned with Better UI/UX -->
<section class="profile-section py-5">
    <div class="container">
        <!-- Page Header -->
        <div class="profile-header mb-4">
            <div class="row align-items-center">
                <div class="col-lg-8">
                    <h1 class="display-5 fw-bold"><?php echo $lang_data['my_profile']; ?></h1>
                    <p class="text-muted"><?php echo isset($lang_data['profile_subtitle']) ? $lang_data['profile_subtitle'] : 'Manage your account, bookings, and content purchases'; ?></p>
                </div>
                <div class="col-lg-4 text-lg-end">
                    <a href="book-studio.php" class="btn btn-primary me-2">
                        <i class="fas fa-calendar-plus me-2"></i><?php echo $lang_data['book_studio']; ?>
                    </a>
                    <a href="shows.php" class="btn btn-outline-primary">
                        <i class="fas fa-film me-2"></i><?php echo $lang_data['shows']; ?>
                    </a>
                </div>
            </div>
        </div>

        <div class="row">
            <!-- Sidebar Navigation -->
            <div class="col-lg-3 mb-4">
                <div class="card profile-sidebar border-0 shadow-sm">
                    <div class="card-body">
                        <div class="text-center mb-4">
                            <div class="user-avatar mb-3">
                                <span class="avatar-circle bg-primary">
                                    <?php echo strtoupper(substr($currentUser['name'], 0, 1)); ?>
                                </span>
                            </div>
                            <h5 class="fw-bold"><?php echo htmlspecialchars($currentUser['name']); ?></h5>
                            <p class="text-muted mb-0"><?php echo htmlspecialchars($currentUser['email']); ?></p>
                            <div class="mt-2">
                                <span class="badge bg-success"><?php echo $lang_data['active']; ?></span>
                            </div>
                        </div>
                        
                        <ul class="nav flex-column nav-pills nav-tabs-profile" id="profileTabs" role="tablist">
                            <li class="nav-item" role="presentation">
                                <button class="nav-link active d-flex align-items-center" id="profile-tab" data-bs-toggle="tab" data-bs-target="#profile-tab-pane" type="button" role="tab">
                                    <i class="fas fa-user me-3"></i> <?php echo $lang_data['profile_information']; ?>
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link d-flex align-items-center" id="bookings-tab" data-bs-toggle="tab" data-bs-target="#bookings-tab-pane" type="button" role="tab">
                                    <i class="fas fa-calendar-check me-3"></i> <?php echo $lang_data['my_bookings']; ?>
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link d-flex align-items-center" id="content-tab" data-bs-toggle="tab" data-bs-target="#content-tab-pane" type="button" role="tab">
                                    <i class="fas fa-play-circle me-3"></i> <?php echo $lang_data['my_content']; ?>
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link d-flex align-items-center" id="password-tab" data-bs-toggle="tab" data-bs-target="#password-tab-pane" type="button" role="tab">
                                    <i class="fas fa-lock me-3"></i> <?php echo $lang_data['change_password']; ?>
                                </button>
                            </li>
                            <li class="nav-item mt-3">
                                <a class="btn btn-outline-danger w-100" href="logout.php">
                                    <i class="fas fa-sign-out-alt me-2"></i> <?php echo $lang_data['logout']; ?>
                                </a>
                            </li>
                        </ul>
                        
                        <!-- User Stats -->
                        <div class="user-stats mt-4 pt-4 border-top">
                            <h6 class="text-uppercase text-muted mb-3"><?php echo $lang_data['member_since']; ?></h6>
                            <p class="mb-3">
                                <i class="far fa-calendar-alt me-2 text-primary"></i>
                                <?php echo date('F j, Y', strtotime($currentUser['created_at'])); ?>
                            </p>
                            
                            <h6 class="text-uppercase text-muted mb-3"><?php echo $lang_data['statistics']; ?></h6>
                            <div class="d-flex justify-content-between mb-2">
                                <span><?php echo $lang_data['bookings']; ?></span>
                                <span class="badge bg-primary rounded-pill"><?php echo count($bookings); ?></span>
                            </div>
                            <div class="d-flex justify-content-between">
                                <span><?php echo $lang_data['purchased_content']; ?></span>
                                <span class="badge bg-primary rounded-pill"><?php echo count($purchases); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Main Content Area -->
            <div class="col-lg-9">
                <div class="tab-content" id="profileTabsContent">
                    <!-- Profile Information Tab -->
                    <div class="tab-pane fade show active" id="profile-tab-pane" role="tabpanel" aria-labelledby="profile-tab" tabindex="0">
                        <div class="card border-0 shadow-sm">
                            <div class="card-header bg-white py-3">
                                <h5 class="card-title mb-0"><?php echo $lang_data['profile_information']; ?></h5>
                            </div>
                            <div class="card-body">
                                <?php if ($updateSuccess): ?>
                                    <div class="alert alert-success alert-dismissible fade show">
                                        <i class="fas fa-check-circle me-2"></i>
                                        <?php echo $lang_data['profile_updated']; ?>
                                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($updateError)): ?>
                                    <div class="alert alert-danger alert-dismissible fade show">
                                        <i class="fas fa-exclamation-circle me-2"></i>
                                        <?php echo $updateError; ?>
                                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                                    </div>
                                <?php endif; ?>
                                
                                <form action="profile.php" method="post">
                                    <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                                    <input type="hidden" name="update_profile" value="1">
                                    
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="name" class="form-label"><?php echo $lang_data['full_name']; ?></label>
                                            <div class="input-group">
                                                <span class="input-group-text"><i class="fas fa-user"></i></span>
                                                <input type="text" class="form-control" id="name" name="name" value="<?php echo htmlspecialchars($currentUser['name']); ?>" required>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-6 mb-3">
                                            <label for="email" class="form-label"><?php echo $lang_data['email_address']; ?></label>
                                            <div class="input-group">
                                                <span class="input-group-text"><i class="fas fa-envelope"></i></span>
                                                <input type="email" class="form-control" id="email" name="email" value="<?php echo htmlspecialchars($currentUser['email']); ?>" required>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="phone" class="form-label"><?php echo $lang_data['phone_number']; ?></label>
                                        <div class="input-group">
                                            <span class="input-group-text"><i class="fas fa-phone"></i></span>
                                            <input type="tel" class="form-control" id="phone" name="phone" value="<?php echo isset($currentUser['phone']) ? htmlspecialchars($currentUser['phone']) : ''; ?>">
                                        </div>
                                    </div>
                                    
                                    <div class="d-grid gap-2 d-md-flex justify-content-md-end mt-4">
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-save me-2"></i><?php echo $lang_data['update_profile']; ?>
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Bookings Tab -->
                    <div class="tab-pane fade" id="bookings-tab-pane" role="tabpanel" aria-labelledby="bookings-tab" tabindex="0">
                        <div class="card border-0 shadow-sm">
                            <div class="card-header bg-white py-3 d-flex justify-content-between align-items-center">
                                <h5 class="card-title mb-0"><?php echo $lang_data['my_bookings']; ?></h5>
                                <a href="book-studio.php" class="btn btn-sm btn-primary">
                                    <i class="fas fa-plus me-1"></i> <?php echo $lang_data['book_now']; ?>
                                </a>
                            </div>
                            <div class="card-body">
                                <?php if (empty($bookings)): ?>
                                    <div class="text-center py-5">
                                        <div class="empty-state-icon mb-3">
                                            <i class="fas fa-calendar-times fa-5x text-muted"></i>
                                        </div>
                                        <h5 class="fw-bold"><?php echo $lang_data['no_bookings_yet']; ?></h5>
                                        <p class="text-muted"><?php echo isset($lang_data['no_bookings_message']) ? $lang_data['no_bookings_message'] : 'Ready to create something amazing? Book our studio for your next production.'; ?></p>
                                        <a href="book-studio.php" class="btn btn-primary mt-2">
                                            <i class="fas fa-calendar-plus me-2"></i><?php echo $lang_data['book_studio']; ?>
                                        </a>
                                    </div>
                                <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover align-middle">
                                            <thead class="table-light">
                                                <tr>
                                                    <th><?php echo $lang_data['reference']; ?></th>
                                                    <th><?php echo $lang_data['service']; ?></th>
                                                    <th><?php echo $lang_data['date_time']; ?></th>
                                                    <th><?php echo $lang_data['status']; ?></th>
                                                    <th><?php echo $lang_data['total']; ?></th>
                                                    <th><?php echo $lang_data['actions']; ?></th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($bookings as $booking): ?>
                                                    <tr>
                                                        <td>
                                                            <span class="fw-medium"><?php echo htmlspecialchars($booking['booking_reference']); ?></span>
                                                        </td>
                                                        <td><?php echo htmlspecialchars($booking['service_name']); ?></td>
                                                        <td>
                                                            <div>
                                                                <i class="far fa-calendar-alt me-1 text-muted"></i>
                                                                <?php echo date('M j, Y', strtotime($booking['booking_date'])); ?>
                                                            </div>
                                                            <div class="small text-muted">
                                                                <i class="far fa-clock me-1"></i>
                                                                <?php echo date('g:i A', strtotime($booking['start_time'])) . ' - ' . 
                                                                         date('g:i A', strtotime($booking['end_time'])); ?>
                                                            </div>
                                                        </td>
                                                        <td>
                                                            <?php
                                                            $statusClass = '';
                                                            $statusIcon = '';
                                                            switch ($booking['status']) {
                                                                case 'pending':
                                                                    $statusClass = 'bg-warning';
                                                                    $statusIcon = 'clock';
                                                                    break;
                                                                case 'confirmed':
                                                                    $statusClass = 'bg-success';
                                                                    $statusIcon = 'check-circle';
                                                                    break;
                                                                case 'completed':
                                                                    $statusClass = 'bg-info';
                                                                    $statusIcon = 'flag-checkered';
                                                                    break;
                                                                case 'canceled':
                                                                    $statusClass = 'bg-danger';
                                                                    $statusIcon = 'times-circle';
                                                                    break;
                                                            }
                                                            ?>
                                                            <span class="badge <?php echo $statusClass; ?>">
                                                                <i class="fas fa-<?php echo $statusIcon; ?> me-1"></i>
                                                                <?php echo isset($lang_data[$booking['status']]) ? $lang_data[$booking['status']] : $booking['status']; ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <span class="fw-bold">
                                                                <?php echo isset($booking['total_amount']) ? number_format($booking['total_amount']) . ' ' . CURRENCY : 
                                                                (isset($booking['amount']) ? number_format($booking['amount']) . ' ' . CURRENCY : ''); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <div class="btn-group">
                                                                <a href="booking-details.php?id=<?php echo $booking['id']; ?>" class="btn btn-sm btn-outline-primary" data-bs-toggle="tooltip" title="<?php echo $lang_data['view_details']; ?>">
                                                                    <i class="fas fa-eye"></i>
                                                                </a>
                                                                <?php if ($booking['status'] === 'pending'): ?>
                                                                    <a href="cancel-booking.php?id=<?php echo $booking['id']; ?>" class="btn btn-sm btn-outline-danger" data-bs-toggle="tooltip" title="<?php echo $lang_data['cancel_booking']; ?>">
                                                                        <i class="fas fa-times"></i>
                                                                    </a>
                                                                <?php endif; ?>
                                                            </div>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Content Tab -->
                    <div class="tab-pane fade" id="content-tab-pane" role="tabpanel" aria-labelledby="content-tab" tabindex="0">
                        <div class="card border-0 shadow-sm">
                            <div class="card-header bg-white py-3 d-flex justify-content-between align-items-center">
                                <h5 class="card-title mb-0"><?php echo $lang_data['my_content']; ?></h5>
                                <a href="shows.php" class="btn btn-sm btn-primary">
                                    <i class="fas fa-film me-1"></i> <?php echo $lang_data['browse_shows']; ?>
                                </a>
                            </div>
                            <div class="card-body">
                                <?php if (empty($purchases)): ?>
                                    <div class="text-center py-5">
                                        <div class="empty-state-icon mb-3">
                                            <i class="fas fa-tv fa-5x text-muted"></i>
                                        </div>
                                        <h5 class="fw-bold"><?php echo $lang_data['no_content_yet']; ?></h5>
                                        <p class="text-muted"><?php echo isset($lang_data['no_content_message']) ? $lang_data['no_content_message'] : 'Discover our premium content and shows to enhance your experience.'; ?></p>
                                        <a href="shows.php" class="btn btn-primary mt-2">
                                            <i class="fas fa-film me-2"></i><?php echo $lang_data['view_all_shows']; ?>
                                        </a>
                                    </div>
                                <?php else: ?>
                                    <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
                                        <?php foreach ($purchases as $purchase): ?>
                                            <div class="col">
                                                <div class="card h-100 content-card">
                                                    <div class="content-thumbnail ratio ratio-16x9">
                                                        <img src="<?php echo htmlspecialchars($purchase['thumbnail']); ?>" 
                                                             class="card-img-top object-fit-cover" 
                                                             alt="<?php echo htmlspecialchars($purchase['title']); ?>">
                                                        <div class="content-overlay">
                                                            <a href="watch.php?id=<?php echo $purchase['content_id']; ?>" class="btn btn-play">
                                                                <i class="fas fa-play"></i>
                                                            </a>
                                                        </div>
                                                    </div>
                                                    <div class="card-body">
                                                        <h5 class="card-title"><?php echo htmlspecialchars($purchase['title']); ?></h5>
                                                        <p class="card-text small text-truncate-2">
                                                            <?php echo htmlspecialchars(substr($purchase['description'] ?? '', 0, 100) . '...'); ?>
                                                        </p>
                                                    </div>
                                                    <div class="card-footer bg-white border-top-0">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <small class="text-muted">
                                                                <i class="far fa-calendar-alt me-1"></i>
                                                                <?php echo $lang_data['purchased_on']; ?>: <?php echo date('M j, Y', strtotime($purchase['created_at'])); ?>
                                                            </small>
                                                            <a href="watch.php?id=<?php echo $purchase['content_id']; ?>" class="btn btn-sm btn-primary">
                                                                <?php echo $lang_data['watch_now']; ?>
                                                            </a>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Change Password Tab -->
                    <div class="tab-pane fade" id="password-tab-pane" role="tabpanel" aria-labelledby="password-tab" tabindex="0">
                        <div class="card border-0 shadow-sm">
                            <div class="card-header bg-white py-3">
                                <h5 class="card-title mb-0"><?php echo $lang_data['change_password']; ?></h5>
                            </div>
                            <div class="card-body">
                                <?php if ($passwordSuccess): ?>
                                    <div class="alert alert-success alert-dismissible fade show">
                                        <i class="fas fa-check-circle me-2"></i>
                                        <?php echo $lang_data['password_changed']; ?>
                                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($passwordError)): ?>
                                    <div class="alert alert-danger alert-dismissible fade show">
                                        <i class="fas fa-exclamation-circle me-2"></i>
                                        <?php echo $passwordError; ?>
                                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                                    </div>
                                <?php endif; ?>
                                
                                <form action="profile.php" method="post">
                                    <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                                    <input type="hidden" name="change_password" value="1">
                                    
                                    <div class="mb-3">
                                        <label for="current_password" class="form-label"><?php echo $lang_data['current_password']; ?></label>
                                        <div class="input-group">
                                            <span class="input-group-text"><i class="fas fa-lock"></i></span>
                                            <input type="password" class="form-control" id="current_password" name="current_password" required>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="new_password" class="form-label"><?php echo $lang_data['new_password']; ?></label>
                                        <div class="input-group">
                                            <span class="input-group-text"><i class="fas fa-key"></i></span>
                                            <input type="password" class="form-control" id="new_password" name="new_password" required>
                                        </div>
                                        <div class="form-text">
                                            <i class="fas fa-info-circle me-1"></i>
                                            <?php echo $lang_data['password_min_length']; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="confirm_password" class="form-label"><?php echo $lang_data['confirm_password']; ?></label>
                                        <div class="input-group">
                                            <span class="input-group-text"><i class="fas fa-check"></i></span>
                                            <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                                        </div>
                                    </div>
                                    
                                    <div class="d-grid gap-2 d-md-flex justify-content-md-end mt-4">
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-key me-2"></i><?php echo $lang_data['change_password']; ?>
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Add required CSS -->
<style>
    /* Profile Section Styles */
    .profile-section {
        background-color: #f8f9fa;
        min-height: 100vh;
    }
    
    .profile-header h1 {
        color: #333;
    }
    
    /* User Avatar Styles */
    .avatar-circle {
        width: 80px;
        height: 80px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 32px;
        font-weight: bold;
        color: white;
        margin: 0 auto;
    }
    
    /* Nav Pills Custom Styling */
    .nav-tabs-profile {
        border-radius: 0.5rem;
        overflow: hidden;
    }
    
    .nav-tabs-profile .nav-link {
        padding: 0.75rem 1rem;
        border: none;
        color: #555;
        border-radius: 0;
        margin-bottom: 5px;
        transition: all 0.2s ease;
    }
    
    .nav-tabs-profile .nav-link.active,
    .nav-tabs-profile .nav-link:hover {
        background-color: rgba(var(--bs-primary-rgb), 0.1);
        color: var(--bs-primary);
        border-left: 3px solid var(--bs-primary);
    }
    
    /* Empty State Icons */
    .empty-state-icon {
        opacity: 0.5;
    }
    
    /* Content Thumbnail with Overlay */
    .content-card {
        transition: all 0.3s ease;
        border: none;
        overflow: hidden;
    }
    
    .content-card:hover {
        box-shadow: 0 .5rem 1rem rgba(0,0,0,.15)!important;
        transform: translateY(-5px);
    }
    
    .content-thumbnail {
        position: relative;
        overflow: hidden;
    }
    
    .content-overlay {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.3);
        display: flex;
        align-items: center;
        justify-content: center;
        opacity: 0;
        transition: opacity 0.3s ease;
    }
    
    .content-thumbnail:hover .content-overlay {
        opacity: 1;
    }
    
    .btn-play {
        width: 50px;
        height: 50px;
        border-radius: 50%;
        background-color: rgba(255, 255, 255, 0.9);
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--bs-primary);
        font-size: 1.2rem;
        transition: all 0.3s ease;
    }
    
    .btn-play:hover {
        background-color: var(--bs-primary);
        color: white;
        transform: scale(1.1);
    }
    
    /* Text truncation for multi-line */
    .text-truncate-2 {
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
        text-overflow: ellipsis;
    }
</style>

<!-- Initialize tooltips -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize Bootstrap tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function(tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
});
</script>

<?php
// Include footer
include 'includes/footer.php';
?>