<?php
/**
 * Process Content Payment
 * Handles payments for pay-per-view content
 */

// Include configuration
require_once 'includes/config.php';

// Include required classes
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/content.php';
require_once 'includes/payment.php';

// Initialize classes
$auth = new Auth();
$contentManager = new Content();
$paymentManager = new Payment();

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "languages/{$lang}.php";

// Check if user is logged in
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

// Get current user
$currentUser = $auth->getCurrentUser();
$userId = $currentUser['id'];

// Check if form is submitted
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: index.php');
    exit;
}

// Verify CSRF token
if (!isset($_POST['csrf_token']) || !$auth->verifyCsrfToken($_POST['csrf_token'])) {
    $_SESSION['payment_error'] = $lang_data['invalid_token'];
    header('Location: index.php');
    exit;
}

// Validate required fields
$required_fields = ['content_id', 'payment_method', 'amount'];
foreach ($required_fields as $field) {
    if (!isset($_POST[$field]) || empty($_POST[$field])) {
        $_SESSION['payment_error'] = $lang_data['all_fields_required'];
        header('Location: watch.php?id=' . ($_POST['content_id'] ?? ''));
        exit;
    }
}

// Get content details
$contentId = (int)$_POST['content_id'];
$contentItem = $contentManager->getContentById($contentId);

// Validate content exists and is published
if (!$contentItem || $contentItem['status'] !== 'published') {
    $_SESSION['payment_error'] = $lang_data['content_not_found'];
    header('Location: index.php');
    exit;
}

// Check if user already has access
if ($paymentManager->hasContentAccess($userId, $contentId)) {
    // Redirect to watch page - already purchased
    header('Location: watch.php?id=' . $contentId);
    exit;
}

// Validate payment amount matches content price
$amount = (float)$_POST['amount'];
if ($amount != $contentItem['price']) {
    $_SESSION['payment_error'] = $lang_data['invalid_amount'];
    header('Location: watch.php?id=' . $contentId);
    exit;
}

// Validate payment method
$paymentMethod = $_POST['payment_method'];
if (!in_array($paymentMethod, ['flooz', 'tmoney'])) {
    $_SESSION['payment_error'] = $lang_data['invalid_payment_method'];
    header('Location: watch.php?id=' . $contentId);
    exit;
}

// Store payment information in session
$_SESSION['pending_content_payment'] = [
    'content_id' => $contentId,
    'amount' => $amount,
    'payment_method' => $paymentMethod
];

// Redirect to content payment page
header('Location: content-payment.php');
exit;
