<?php
/**
 * Process Booking Form
 * Handles the form submission for studio bookings
 */

// Include configuration
require_once 'includes/config.php';

// Include required classes
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/booking.php';
require_once 'includes/payment.php';

// Initialize classes
$auth = new Auth();
$bookingManager = new Booking();
$paymentManager = new Payment();

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "languages/{$lang}.php";

// Check if form is submitted
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: index.php#booking');
    exit;
}

// Verify CSRF token
if (!isset($_POST['csrf_token']) || !$auth->verifyCsrfToken($_POST['csrf_token'])) {
    $_SESSION['booking_error'] = $lang_data['invalid_token'];
    header('Location: index.php#booking');
    exit;
}

// Validate required fields
$required_fields = ['name', 'email', 'phone', 'date', 'time_slot', 'service_id', 'payment_method'];
foreach ($required_fields as $field) {
    if (!isset($_POST[$field]) || empty($_POST[$field])) {
        $_SESSION['booking_error'] = $lang_data['all_fields_required'];
        header('Location: index.php#booking');
        exit;
    }
}

// Validate email
if (!filter_var($_POST['email'], FILTER_VALIDATE_EMAIL)) {
    $_SESSION['booking_error'] = $lang_data['invalid_email'];
    header('Location: index.php#booking');
    exit;
}

// Validate date (must be future date)
$bookingDate = $_POST['date'];
if (strtotime($bookingDate) < strtotime(date('Y-m-d'))) {
    $_SESSION['booking_error'] = $lang_data['invalid_date'];
    header('Location: index.php#booking');
    exit;
}

// Validate time slot is available
$timeSlot = $_POST['time_slot'];
list($startTime, $endTime) = explode(' - ', $timeSlot);

if (!$bookingManager->isTimeSlotAvailable($bookingDate, $startTime, $endTime)) {
    $_SESSION['booking_error'] = $lang_data['slot_unavailable'];
    header('Location: index.php#booking');
    exit;
}

// Validate service exists
$serviceId = (int)$_POST['service_id'];
$db = Database::getInstance();
$service = $db->getRow("SELECT * FROM services WHERE id = ?", [$serviceId]);

if (!$service) {
    $_SESSION['booking_error'] = $lang_data['invalid_service'];
    header('Location: index.php#booking');
    exit;
}

// Get current user or create guest account
$userId = null;
if ($auth->isLoggedIn()) {
    $currentUser = $auth->getCurrentUser();
    $userId = $currentUser['id'];
} else {
    // Check if user exists with this email
    $query = "SELECT id FROM users WHERE email = ?";
    $userId = $db->getValue($query, [$_POST['email']]);
    
    // Create user if not exists
    if (!$userId) {
        $userData = [
            'name' => $_POST['name'],
            'email' => $_POST['email'],
            'phone' => $_POST['phone'],
            'role' => 'customer',
            'status' => 'active',
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $userId = $db->insert('users', $userData);
    }
}

// Prepare booking data
$bookingData = [
    'user_id' => $userId,
    'service_id' => $serviceId,
    'date' => $bookingDate,
    'time_slot' => $timeSlot,
    'notes' => $_POST['notes'] ?? '',
    'amount' => $service['price'],
    'name' => $_POST['name'],
    'email' => $_POST['email'],
    'phone' => $_POST['phone']
];

// Create booking
$bookingId = $bookingManager->createBooking($bookingData);

if (!$bookingId) {
    $_SESSION['booking_error'] = $lang_data['booking_failed'];
    header('Location: index.php#booking');
    exit;
}

// Get the created booking
$booking = $bookingManager->getBookingById($bookingId);

// Store booking information for payment
$_SESSION['pending_booking'] = [
    'id' => $bookingId,
    'reference' => $booking['booking_reference'],
    'amount' => $booking['amount'],
    'payment_method' => $_POST['payment_method']
];

// Redirect to payment page
header('Location: payment.php');
exit;
