<?php
/**
 * Payment Processing Page
 * Handles the payment for bookings
 */

// Include configuration
require_once 'includes/config.php';

// Include required classes
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/booking.php';

// Initialize classes
$auth = new Auth();
$bookingManager = new Booking();

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "languages/{$lang}.php";

// Check if we have pending booking
if (!isset($_SESSION['pending_booking']) || empty($_SESSION['pending_booking'])) {
    header('Location: index.php');
    exit;
}

$pendingBooking = $_SESSION['pending_booking'];
$bookingId = $pendingBooking['id'];
$bookingReference = $pendingBooking['reference'];
$amount = $pendingBooking['amount'];
$paymentMethod = $pendingBooking['payment_method'];

// Get booking details
$booking = $bookingManager->getBookingById($bookingId);

if (!$booking) {
    // If booking not found, clear session and redirect
    unset($_SESSION['pending_booking']);
    header('Location: index.php');
    exit;
}

// Determine payment provider details
$paymentPhone = '';
$paymentInstructions = '';
$paymentLogo = '';

if ($paymentMethod === 'flooz') {
    $paymentPhone = FLOOZ_PHONE;
    $paymentLogo = 'assets/images/flooz.png';
    $paymentInstructions = $lang_data['flooz_instructions'];
} elseif ($paymentMethod === 'tmoney') {
    $paymentPhone = TMONEY_PHONE;
    $paymentLogo = 'assets/images/tmoney.png';
    $paymentInstructions = $lang_data['tmoney_instructions'];
}

// Handle payment form submission
$transactionError = '';
$paymentSuccess = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['transaction_id'])) {
    // Verify CSRF token
    if (!isset($_POST['csrf_token']) || !$auth->verifyCsrfToken($_POST['csrf_token'])) {
        $transactionError = $lang_data['invalid_token'];
    } else {
        $transactionId = trim($_POST['transaction_id']);
        
        // Basic validation: Transaction ID should be at least 8 characters
        if (strlen($transactionId) < 8) {
            $transactionError = $lang_data['invalid_transaction'];
        } else {
            // In a real application, you would verify the transaction with the payment provider
            // For this example, we'll simulate successful verification
            
            // Update booking payment status
            $paymentData = [
                'booking_id' => $bookingId,
                'amount' => $amount,
                'payment_method' => $paymentMethod,
                'transaction_id' => $transactionId,
                'status' => 'completed',
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $db = Database::getInstance();
            $paymentId = $db->insert('payments', $paymentData);
            
            if ($paymentId) {
                // Update booking status
                $bookingManager->updatePaymentStatus($bookingId, 'paid', $transactionId);
                $bookingManager->updateBookingStatus($bookingId, 'confirmed');
                
                // Send confirmation SMS
                $bookingManager->sendBookingConfirmationSMS($bookingId);
                
                // Payment successful
                $paymentSuccess = true;
                
                // Clear pending booking from session
                unset($_SESSION['pending_booking']);
            } else {
                $transactionError = $lang_data['payment_failed'];
            }
        }
    }
}

// Get page title
$pageTitle = $lang_data['payment_title'] . ' - ' . SITE_NAME;

// Include header
include 'includes/header.php';
?>

<section class="payment-section py-5">
    <div class="container">
        <?php if ($paymentSuccess): ?>
            <!-- Payment Success -->
            <div class="card payment-success-card">
                <div class="card-body text-center">
                    <div class="mb-4">
                        <i class="fas fa-check-circle text-success" style="font-size: 5rem;"></i>
                    </div>
                    <h2 class="card-title mb-4"><?php echo $lang_data['payment_successful']; ?></h2>
                    <p><?php echo $lang_data['booking_confirmed']; ?></p>
                    
                    <div class="booking-details bg-light p-4 rounded my-4">
                        <div class="row">
                            <div class="col-md-6 text-md-end"><strong><?php echo $lang_data['booking_reference']; ?>:</strong></div>
                            <div class="col-md-6 text-md-start"><?php echo $bookingReference; ?></div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 text-md-end"><strong><?php echo $lang_data['service']; ?>:</strong></div>
                            <div class="col-md-6 text-md-start"><?php echo $booking['service_name']; ?></div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 text-md-end"><strong><?php echo $lang_data['date']; ?>:</strong></div>
                            <div class="col-md-6 text-md-start"><?php echo date('F j, Y', strtotime($booking['booking_date'])); ?></div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 text-md-end"><strong><?php echo $lang_data['time']; ?>:</strong></div>
                            <div class="col-md-6 text-md-start">
                                <?php 
                                echo date('g:i A', strtotime($booking['start_time'])) . ' - ' . 
                                     date('g:i A', strtotime($booking['end_time'])); 
                                ?>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 text-md-end"><strong><?php echo $lang_data['amount']; ?>:</strong></div>
                            <div class="col-md-6 text-md-start"><?php echo number_format($amount) . ' ' . CURRENCY; ?></div>
                        </div>
                    </div>
                    
                    <p><?php echo $lang_data['confirmation_sent']; ?></p>
                    
                    <div class="mt-4">
                        <a href="index.php" class="btn btn-primary"><?php echo $lang_data['back_home']; ?></a>
                        <a href="my-bookings.php" class="btn btn-outline-primary"><?php echo $lang_data['my_bookings']; ?></a>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <!-- Payment Form -->
            <div class="row justify-content-center">
                <div class="col-md-8">
                    <div class="card">
                        <div class="card-header bg-primary text-white">
                            <h3 class="card-title mb-0"><?php echo $lang_data['complete_payment']; ?></h3>
                        </div>
                        <div class="card-body">
                            <?php if (!empty($transactionError)): ?>
                                <div class="alert alert-danger">
                                    <?php echo $transactionError; ?>
                                </div>
                            <?php endif; ?>
                            
                            <div class="payment-details mb-4">
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['booking_reference']; ?>:</div>
                                    <div class="col-md-8"><?php echo $bookingReference; ?></div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['service']; ?>:</div>
                                    <div class="col-md-8"><?php echo $booking['service_name']; ?></div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['date']; ?>:</div>
                                    <div class="col-md-8"><?php echo date('F j, Y', strtotime($booking['booking_date'])); ?></div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['time']; ?>:</div>
                                    <div class="col-md-8">
                                        <?php 
                                        echo date('g:i A', strtotime($booking['start_time'])) . ' - ' . 
                                             date('g:i A', strtotime($booking['end_time'])); 
                                        ?>
                                    </div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['amount']; ?>:</div>
                                    <div class="col-md-8 fw-bold text-primary">
                                        <?php echo number_format($amount) . ' ' . CURRENCY; ?>
                                    </div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['payment_method']; ?>:</div>
                                    <div class="col-md-8">
                                        <div class="d-flex align-items-center">
                                            <img src="<?php echo $paymentLogo; ?>" alt="<?php echo ucfirst($paymentMethod); ?>" 
                                                 style="height: 30px; margin-right: 10px;">
                                            <?php echo ucfirst($paymentMethod) . ' ' . $lang_data['mobile_money']; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="payment-instructions alert alert-info">
                                <h5><?php echo $lang_data['payment_instructions']; ?></h5>
                                <ol>
                                    <li><?php echo str_replace('{PHONE}', $paymentPhone, $paymentInstructions); ?></li>
                                    <li><?php echo $lang_data['enter_amount'] . ': ' . number_format($amount) . ' ' . CURRENCY; ?></li>
                                    <li><?php echo $lang_data['reference_info'] . ': ' . $bookingReference; ?></li>
                                    <li><?php echo $lang_data['confirm_payment']; ?></li>
                                    <li><?php echo $lang_data['enter_transaction_below']; ?></li>
                                </ol>
                                
                                <div class="text-center mt-3">
                                    <div class="payment-number">
                                        <?php echo $paymentPhone; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <form action="payment.php" method="post">
                                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                                
                                <div class="mb-3">
                                    <label for="transaction_id" class="form-label"><?php echo $lang_data['transaction_id']; ?></label>
                                    <input type="text" class="form-control" id="transaction_id" name="transaction_id" required>
                                    <div class="form-text"><?php echo $lang_data['transaction_help']; ?></div>
                                </div>
                                
                                <div class="d-grid gap-2">
                                    <button type="submit" class="btn btn-primary btn-lg">
                                        <?php echo $lang_data['confirm_payment_button']; ?>
                                    </button>
                                    <a href="cancel-booking.php?id=<?php echo $bookingId; ?>&token=<?php echo $auth->generateCsrfToken(); ?>" 
                                       class="btn btn-outline-danger">
                                        <?php echo $lang_data['cancel_booking']; ?>
                                    </a>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</section>

<?php
// Include footer
include 'includes/footer.php';
?>
