<?php
/**
 * User Login Page - 100% Error Protected with Session Fix
 */

// For debugging (can be removed in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// REMOVED session_start() - Let config.php handle session initialization

// Try to include configuration
if (file_exists('includes/config.php')) {
    require_once 'includes/config.php';
} else {
    die("Configuration file not found.");
}

// Try to include required classes
if (file_exists('includes/db.php') && file_exists('includes/auth.php')) {
    require_once 'includes/db.php';
    require_once 'includes/auth.php';
} else {
    die("Required class files not found.");
}

// Initialize classes
try {
    $auth = new Auth();
} catch (Exception $e) {
    die("Failed to initialize authentication: " . $e->getMessage());
}

// Check if user is already logged in
if ($auth->isLoggedIn()) {
    // Redirect to home page or requested page
    $redirect = isset($_GET['redirect']) ? $_GET['redirect'] : 'index.php';
    header('Location: ' . $redirect);
    exit;
}

// Process login form
$loginError = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // SAFELY get POST variables
    $email = isset($_POST['email']) ? trim($_POST['email']) : '';
    $password = isset($_POST['password']) ? $_POST['password'] : '';  // No trim() for password
    $rememberMe = isset($_POST['remember_me']) && $_POST['remember_me'] == '1';
    $csrf_token = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';

    // Verify CSRF token
    if (empty($csrf_token) || !$auth->verifyCsrfToken($csrf_token)) {
        $loginError = 'Invalid security token. Please try again.';
    } else {
        // Validate email and password
        if (empty($email) || empty($password)) {
            $loginError = 'Please enter both email and password.';
        } else {
            try {
                // Attempt login
                $result = $auth->login($email, $password);
                
                if (isset($result['success']) && $result['success']) {
                    // Set remember me cookie if requested
                    if ($rememberMe && isset($result['user']) && isset($result['user']['id'])) {
                        try {
                            // Generate token and store in database
                            $token = bin2hex(random_bytes(32));
                            $userId = $result['user']['id'];
                            $expiresAt = date('Y-m-d H:i:s', strtotime('+30 days'));
                            
                            $db = Database::getInstance();
                            
                            // Check if remember_tokens table exists
                            $tableExists = $db->getValue("SHOW TABLES LIKE 'remember_tokens'");
                            
                            if ($tableExists) {
                                $db->query(
                                    "INSERT INTO remember_tokens (user_id, token, expires_at, created_at) 
                                    VALUES (?, ?, ?, NOW())",
                                    [$userId, $token, $expiresAt]
                                );
                                
                                // Set cookie
                                setcookie('remember_token', $token, time() + (86400 * 30), '/', '', true, true);
                            }
                        } catch (Exception $e) {
                            // Silently fail if remember tokens don't work
                            error_log("Remember token error: " . $e->getMessage());
                        }
                    }
                    
                    // Redirect after successful login
                    $redirect = isset($_GET['redirect']) ? $_GET['redirect'] : 'index.php';
                    header('Location: ' . $redirect);
                    exit;
                } else {
                    $loginError = isset($result['message']) ? $result['message'] : 'Login failed';
                }
            } catch (Exception $e) {
                $loginError = 'An error occurred during login: ' . $e->getMessage();
                error_log("Login error: " . $e->getMessage());
            }
        }
    }
}

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : (defined('DEFAULT_LANGUAGE') ? DEFAULT_LANGUAGE : 'en');
$langFilePath = "languages/{$lang}.php";

if (file_exists($langFilePath)) {
    require_once $langFilePath;
} else {
    // Fallback language data
    $lang_data = [
        'login_title' => 'Login',
        'welcome_back' => 'Welcome Back!',
        'login_message' => 'Login to access your account.',
        'email_address' => 'Email Address',
        'password' => 'Password',
        'remember_me' => 'Remember Me',
        'login' => 'Login',
        'forgot_password' => 'Forgot Password?',
        'dont_have_account' => 'Don\'t have an account?',
        'create_account' => 'Create Account'
    ];
}

// Set page title
$pageTitle = (isset($lang_data['login_title']) ? $lang_data['login_title'] : 'Login') . ' - ' . (defined('SITE_NAME') ? SITE_NAME : 'Web TV Studio');

// Try to include header
$headerIncluded = @include 'includes/header.php';

// If header failed, create a basic one
if (!$headerIncluded) {
    echo '<!DOCTYPE html>';
    echo '<html lang="' . $lang . '">';
    echo '<head>';
    echo '<meta charset="UTF-8">';
    echo '<meta name="viewport" content="width=device-width, initial-scale=1.0">';
    echo '<title>' . $pageTitle . '</title>';
    echo '<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">';
    echo '</head>';
    echo '<body>';
}
?>

<section class="login-section py-5">
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-body p-4">
                        <div class="text-center mb-4">
                            <h2><?php echo isset($lang_data['welcome_back']) ? $lang_data['welcome_back'] : 'Welcome Back!'; ?></h2>
                            <p class="text-muted"><?php echo isset($lang_data['login_message']) ? $lang_data['login_message'] : 'Login to access your account.'; ?></p>
                        </div>
                        
                        <?php if (!empty($loginError)): ?>
                            <div class="alert alert-danger">
                                <?php echo $loginError; ?>
                            </div>
                        <?php endif; ?>
                        
                        <form action="login.php<?php echo isset($_GET['redirect']) ? '?redirect=' . urlencode($_GET['redirect']) : ''; ?>" method="post">
                            <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                            
                            <div class="mb-3">
                                <label for="email" class="form-label"><?php echo isset($lang_data['email_address']) ? $lang_data['email_address'] : 'Email Address'; ?></label>
                                <input type="email" class="form-control" id="email" name="email" required 
                                       value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>">
                            </div>
                            
                            <div class="mb-3">
                                <label for="password" class="form-label"><?php echo isset($lang_data['password']) ? $lang_data['password'] : 'Password'; ?></label>
                                <input type="password" class="form-control" id="password" name="password" required>
                            </div>
                            
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="remember_me" name="remember_me" value="1"
                                       <?php echo isset($_POST['remember_me']) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="remember_me"><?php echo isset($lang_data['remember_me']) ? $lang_data['remember_me'] : 'Remember Me'; ?></label>
                            </div>
                            
                            <div class="d-grid">
                                <button type="submit" class="btn btn-primary"><?php echo isset($lang_data['login']) ? $lang_data['login'] : 'Login'; ?></button>
                            </div>
                            
                            <div class="text-center mt-3">
                                <a href="forgot-password.php" class="text-decoration-none"><?php echo isset($lang_data['forgot_password']) ? $lang_data['forgot_password'] : 'Forgot Password?'; ?></a>
                            </div>
                        </form>
                        
                        <hr class="my-4">
                        
                        <div class="text-center">
                            <p><?php echo isset($lang_data['dont_have_account']) ? $lang_data['dont_have_account'] : 'Don\'t have an account?'; ?></p>
                            <a href="register.php<?php echo isset($_GET['redirect']) ? '?redirect=' . urlencode($_GET['redirect']) : ''; ?>" 
                               class="btn btn-outline-primary"><?php echo isset($lang_data['create_account']) ? $lang_data['create_account'] : 'Create Account'; ?></a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<?php
// Try to include footer
$footerIncluded = @include 'includes/footer.php';

// If footer failed, create a basic one
if (!$footerIncluded) {
    echo '<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>';
    echo '</body>';
    echo '</html>';
}
?>