<?php
/**
 * Enhanced Web TV Studio Homepage
 */

// Include configuration
require_once 'includes/config.php';

// Include required classes
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/content.php';
require_once 'includes/booking.php';

// Initialize classes
$auth = new Auth();
$content = new Content();
$booking = new Booking();

// Get current user if logged in
$currentUser = $auth->getCurrentUser();

// Get featured content
$featuredContent = $content->getFeaturedContent(6);

// Get available slots for today
$today = date('Y-m-d');
$availableSlots = $booking->getAvailableTimeSlots($today);

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "languages/{$lang}.php";

// Get page title
$pageTitle = SITE_NAME;

// Include header
include 'includes/header.php';
?>

<!-- Preloader -->
<div class="preloader">
    <div class="spinner">
        <div class="bounce1"></div>
        <div class="bounce2"></div>
        <div class="bounce3"></div>
    </div>
</div>

<!-- Hero Section -->
<section class="hero" id="home">
    <div class="container hero-container">
        <div class="hero-content">
            <h1 id="hero-title" class="animate-on-scroll"><?php echo $lang_data['hero_title']; ?></h1>
            <p id="hero-subtitle" class="animate-on-scroll"><?php echo $lang_data['hero_subtitle']; ?></p>
            <div class="hero-buttons animate-on-scroll">
                <a href="#booking" class="btn btn-primary btn-lg"><?php echo $lang_data['book_studio']; ?></a>
                <a href="#shows" class="btn btn-outline-light btn-lg ms-2"><?php echo $lang_data['explore_shows']; ?></a>
            </div>
        </div>
    </div>
</section>

<!-- About Section -->
<section class="section" id="about">
    <div class="container">
        <h2 class="section-title animate-on-scroll"><?php echo $lang_data['about_title']; ?></h2>
        <p class="section-subtitle animate-on-scroll"><?php echo $lang_data['about_subtitle']; ?></p>
        
        <div class="row stagger-animation">
            <div class="col-md-4 animate-on-scroll">
                <div class="feature-box">
                    <i class="fas fa-video"></i>
                    <h3><?php echo $lang_data['professional_equipment']; ?></h3>
                    <p><?php echo $lang_data['equipment_text']; ?></p>
                </div>
            </div>
            <div class="col-md-4 animate-on-scroll">
                <div class="feature-box">
                    <i class="fas fa-users"></i>
                    <h3><?php echo $lang_data['expert_team']; ?></h3>
                    <p><?php echo $lang_data['team_text']; ?></p>
                </div>
            </div>
            <div class="col-md-4 animate-on-scroll">
                <div class="feature-box">
                    <i class="fas fa-globe"></i>
                    <h3><?php echo $lang_data['global_reach']; ?></h3>
                    <p><?php echo $lang_data['reach_text']; ?></p>
                </div>
            </div>
        </div>
        
        <div class="text-center mt-5 animate-on-scroll">
            <a href="#team" class="btn btn-outline-primary">
                <?php echo $lang_data['meet_our_team']; ?> <i class="fas fa-arrow-right ms-1"></i>
            </a>
        </div>
    </div>
</section>

<!-- Shows Section -->
<section class="section bg-light" id="shows">
    <div class="container">
        <h2 class="section-title animate-on-scroll"><?php echo $lang_data['shows_title']; ?></h2>
        <p class="section-subtitle animate-on-scroll"><?php echo $lang_data['shows_subtitle']; ?></p>
        
        <?php if (empty($featuredContent)): ?>
            <div class="alert alert-info animate-on-scroll">
                <?php echo $lang_data['no_shows']; ?>
            </div>
        <?php else: ?>
            <div class="row" id="shows-container">
                <?php foreach ($featuredContent as $index => $item): ?>
                    <div class="col-md-6 col-lg-4 mb-4 animate-on-scroll">
                        <div class="video-card h-100 d-flex flex-column">
                            <div class="video-thumbnail" data-video-id="<?php echo $item['id']; ?>" data-video-url="watch.php?id=<?php echo $item['id']; ?>">
                                <img src="<?php echo htmlspecialchars($item['thumbnail']); ?>" alt="<?php echo htmlspecialchars($item['title']); ?>">
                                <div class="play-button">
                                    <i class="fas fa-play-circle"></i>
                                </div>
                                <?php if (isset($item['price']) && $item['price'] > 0): ?>
                                    <div class="price-badge">
                                        <?php echo number_format($item['price']) . ' ' . CURRENCY; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="video-content flex-grow-1">
                                <h4><?php echo htmlspecialchars($item['title']); ?></h4>
                                <p><?php echo htmlspecialchars(substr($item['description'], 0, 100) . '...'); ?></p>
                                <a href="watch.php?id=<?php echo $item['id']; ?>" class="btn btn-sm btn-primary">
                                    <?php echo $lang_data['watch_now']; ?> <i class="fas fa-arrow-right ms-1"></i>
                                </a>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <div class="text-center mt-5 animate-on-scroll">
                <a href="content.php" class="btn btn-outline-primary">
                    <?php echo $lang_data['view_all_shows']; ?> <i class="fas fa-arrow-right ms-1"></i>
                </a>
            </div>
        <?php endif; ?>
    </div>
</section>

<!-- Services Section -->
<section class="section" id="services">
    <div class="container">
        <h2 class="section-title animate-on-scroll"><?php echo $lang_data['services_title']; ?></h2>
        <p class="section-subtitle animate-on-scroll"><?php echo $lang_data['services_subtitle']; ?></p>
        
        <div class="row">
            <?php
            // Get services from database
            $db = Database::getInstance();
            $query = "SELECT * FROM services WHERE active = 1 ORDER BY name";
            $services = $db->getRows($query);
            
            foreach ($services as $index => $service):
            ?>
                <div class="col-md-6 col-lg-4 mb-4 animate-on-scroll">
                    <div class="service-card h-100">
                        <div class="service-icon">
                            <i class="<?php echo htmlspecialchars($service['icon'] ?? 'fas fa-video'); ?>"></i>
                        </div>
                        <h3><?php echo htmlspecialchars($service['name']); ?></h3>
                        <p><?php echo htmlspecialchars($service['description'] ?? ''); ?></p>
                        <div class="service-price">
                            <?php echo number_format($service['price']) . ' ' . CURRENCY; ?>
                        </div>
                        <a href="#booking" class="btn btn-outline-primary mt-3 book-service-btn" data-service-id="<?php echo $service['id']; ?>">
                            <?php echo $lang_data['book_now']; ?>
                        </a>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>

<!-- Team Section -->
<section class="section bg-light" id="team">
    <div class="container">
        <h2 class="section-title animate-on-scroll"><?php echo $lang_data['team_title']; ?></h2>
        <p class="section-subtitle animate-on-scroll"><?php echo $lang_data['team_subtitle']; ?></p>
        
        <div class="row" id="team-container">
            <!-- Team members will be loaded from JavaScript -->
        </div>
    </div>
</section>

<!-- Booking Section -->
<section class="section" id="booking">
    <div class="container">
        <h2 class="section-title animate-on-scroll"><?php echo $lang_data['booking_title']; ?></h2>
        <p class="section-subtitle animate-on-scroll"><?php echo $lang_data['booking_subtitle']; ?></p>
        
        <ul class="nav nav-tabs animate-on-scroll" id="bookingTabs" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link active" id="booking-form-tab" data-bs-toggle="tab" data-bs-target="#booking-form-content" type="button" role="tab" aria-controls="booking-form-content" aria-selected="true">
                    <i class="fas fa-calendar-check me-2"></i><?php echo $lang_data['make_booking']; ?>
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="calendar-tab" data-bs-toggle="tab" data-bs-target="#calendar-content" type="button" role="tab" aria-controls="calendar-content" aria-selected="false">
                    <i class="fas fa-calendar-alt me-2"></i><?php echo $lang_data['view_availability']; ?>
                </button>
            </li>
        </ul>
        
        <div class="tab-content mt-4" id="bookingTabsContent">
            <div class="tab-pane fade show active animate-on-scroll" id="booking-form-content" role="tabpanel" aria-labelledby="booking-form-tab">
                <div class="row">
                    <div class="col-lg-8 mx-auto">
                        <div class="booking-card">
                            <h3 class="text-center mb-4"><?php echo $lang_data['booking_form']; ?></h3>
                            
                            <?php if (isset($_SESSION['booking_success'])): ?>
                                <div class="alert alert-success">
                                    <?php 
                                        echo $_SESSION['booking_success']; 
                                        unset($_SESSION['booking_success']);
                                    ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (isset($_SESSION['booking_error'])): ?>
                                <div class="alert alert-danger">
                                    <?php 
                                        echo $_SESSION['booking_error']; 
                                        unset($_SESSION['booking_error']);
                                    ?>
                                </div>
                            <?php endif; ?>
                            
                            <form id="booking-form" action="process-booking.php" method="post">
                                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="name" class="form-label required"><?php echo $lang_data['full_name']; ?></label>
                                            <input type="text" class="form-control" id="name" name="name" required 
                                                   value="<?php echo $currentUser ? htmlspecialchars($currentUser['name']) : ''; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="email" class="form-label required"><?php echo $lang_data['email_address']; ?></label>
                                            <input type="email" class="form-control" id="email" name="email" required
                                                   value="<?php echo $currentUser ? htmlspecialchars($currentUser['email']) : ''; ?>">
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="phone" class="form-label required"><?php echo $lang_data['phone_number']; ?></label>
                                            <input type="tel" class="form-control" id="phone" name="phone" required
                                                   value="<?php echo $currentUser ? htmlspecialchars($currentUser['phone']) : ''; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="service" class="form-label required"><?php echo $lang_data['service_type']; ?></label>
                                            <select class="form-control" id="service" name="service_id" required>
                                                <option value=""><?php echo $lang_data['select_service']; ?></option>
                                                <?php
                                                foreach ($services as $service):
                                                ?>
                                                    <option value="<?php echo $service['id']; ?>" data-price="<?php echo $service['price']; ?>">
                                                        <?php echo htmlspecialchars($service['name']); ?> - 
                                                        <?php echo number_format($service['price']) . ' ' . CURRENCY; ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="date" class="form-label required"><?php echo $lang_data['preferred_date']; ?></label>
                                            <input type="date" class="form-control" id="date" name="date" required min="<?php echo date('Y-m-d'); ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="time" class="form-label required"><?php echo $lang_data['preferred_time']; ?></label>
                                            <select class="form-control" id="time" name="time_slot" required>
                                                <option value=""><?php echo $lang_data['select_time']; ?></option>
                                                <?php if (!empty($availableSlots)): ?>
                                                    <?php foreach ($availableSlots as $slot): ?>
                                                        <option value="<?php echo htmlspecialchars($slot['value']); ?>">
                                                            <?php echo htmlspecialchars($slot['display']); ?>
                                                        </option>
                                                    <?php endforeach; ?>
                                                <?php else: ?>
                                                    <option value="" disabled><?php echo $lang_data['no_slots_available']; ?></option>
                                                <?php endif; ?>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="message" class="form-label"><?php echo $lang_data['additional_info']; ?></label>
                                    <textarea class="form-control" id="message" name="notes" rows="3"></textarea>
                                </div>
                                
                                <div class="booking-summary mb-4">
                                    <h4 class="mb-3">
                                        <i class="fas fa-receipt me-2"></i><?php echo $lang_data['booking_summary']; ?>
                                    </h4>
                                    <div class="booking-summary-content p-3">
                                        <div class="d-flex justify-content-between mb-2">
                                            <span><?php echo $lang_data['service']; ?>:</span>
                                            <span id="summary-service">-</span>
                                        </div>
                                        <div class="d-flex justify-content-between mb-2">
                                            <span><?php echo $lang_data['date_and_time']; ?>:</span>
                                            <span id="summary-date-time">-</span>
                                        </div>
                                        <div class="d-flex justify-content-between">
                                            <span class="fw-bold"><?php echo $lang_data['total']; ?>:</span>
                                            <span class="fw-bold" id="booking-price">-</span>
                                        </div>
                                    </div>
                                </div>
                                
                                <h4 class="mt-4 mb-3"><?php echo $lang_data['payment_method']; ?></h4>
                                <div class="payment-options">
                                    <div class="payment-option d-flex align-items-center">
                                        <input type="radio" name="payment_method" id="flooz" value="flooz" required>
                                        <label for="flooz" class="ms-2 d-flex align-items-center flex-grow-1">
                                            <img src="assets/images/flooz.png" alt="Flooz" class="me-2">
                                            Flooz Mobile Money
                                        </label>
                                    </div>
                                    <div class="payment-option d-flex align-items-center mt-2">
                                        <input type="radio" name="payment_method" id="tmoney" value="tmoney">
                                        <label for="tmoney" class="ms-2 d-flex align-items-center flex-grow-1">
                                            <img src="assets/images/tmoney.png" alt="T-Money" class="me-2">
                                            T-Money
                                        </label>
                                    </div>
                                </div>
                                
                                <div class="text-center mt-4">
                                    <button type="submit" class="btn btn-primary btn-lg"><?php echo $lang_data['submit_booking']; ?></button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="tab-pane fade" id="calendar-content" role="tabpanel" aria-labelledby="calendar-tab">
                <div class="row">
                    <div class="col-lg-10 mx-auto">
                        <div class="booking-card">
                            <h3 class="text-center mb-4"><?php echo $lang_data['available_slots']; ?></h3>
                            <div id="calendar"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Testimonials Section -->
<section class="section bg-light" id="testimonials">
    <div class="container">
        <h2 class="section-title animate-on-scroll"><?php echo $lang_data['testimonials_title']; ?></h2>
        <p class="section-subtitle animate-on-scroll"><?php echo $lang_data['testimonials_subtitle']; ?></p>
        
        <div class="row">
            <div class="col-lg-10 mx-auto">
                <div id="testimonialCarousel" class="carousel slide animate-on-scroll" data-bs-ride="carousel">
                    <div class="carousel-inner">
                        <div class="carousel-item active">
                            <div class="testimonial-card">
                                <div class="testimonial-content">
                                    <p>"The quality of production at this studio is exceptional. Professional staff and state-of-the-art equipment made our show a success!"</p>
                                </div>
                                <div class="testimonial-author">
                                    <img src="assets/images/testimonials/client1.jpg" alt="Client 1">
                                    <div>
                                        <h5>James Wilson</h5>
                                        <span>Talk Show Host</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="carousel-item">
                            <div class="testimonial-card">
                                <div class="testimonial-content">
                                    <p>"We recorded our music video here and were blown away by the results. The team's creativity and attention to detail exceeded our expectations."</p>
                                </div>
                                <div class="testimonial-author">
                                    <img src="assets/images/testimonials/client2.jpg" alt="Client 2">
                                    <div>
                                        <h5>Harmony Band</h5>
                                        <span>Music Group</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="carousel-item">
                            <div class="testimonial-card">
                                <div class="testimonial-content">
                                    <p>"The streaming quality for our live event was impeccable. Our audience from around the world had a seamless viewing experience."</p>
                                </div>
                                <div class="testimonial-author">
                                    <img src="assets/images/testimonials/client3.jpg" alt="Client 3">
                                    <div>
                                        <h5>Sarah Johnson</h5>
                                        <span>Event Organizer</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <button class="carousel-control-prev" type="button" data-bs-target="#testimonialCarousel" data-bs-slide="prev">
                        <i class="fas fa-chevron-left"></i>
                        <span class="visually-hidden">Previous</span>
                    </button>
                    <button class="carousel-control-next" type="button" data-bs-target="#testimonialCarousel" data-bs-slide="next">
                        <i class="fas fa-chevron-right"></i>
                        <span class="visually-hidden">Next</span>
                    </button>
                    <div class="carousel-indicators">
                        <button type="button" data-bs-target="#testimonialCarousel" data-bs-slide-to="0" class="active" aria-current="true" aria-label="Slide 1"></button>
                        <button type="button" data-bs-target="#testimonialCarousel" data-bs-slide-to="1" aria-label="Slide 2"></button>
                        <button type="button" data-bs-target="#testimonialCarousel" data-bs-slide-to="2" aria-label="Slide 3"></button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Contact Section -->
<section class="section" id="contact">
    <div class="container">
        <h2 class="section-title animate-on-scroll"><?php echo $lang_data['contact_title']; ?></h2>
        <p class="section-subtitle animate-on-scroll"><?php echo $lang_data['contact_subtitle']; ?></p>
        
        <div class="row">
            <div class="col-lg-6 animate-on-scroll">
                <form id="contact-form" action="process-contact.php" method="post" class="contact-form">
                    <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="contact-name" class="form-label required"><?php echo $lang_data['your_name']; ?></label>
                                <input type="text" class="form-control" id="contact-name" name="name" required
                                       value="<?php echo $currentUser ? htmlspecialchars($currentUser['name']) : ''; ?>">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="contact-email" class="form-label required"><?php echo $lang_data['email_address']; ?></label>
                                <input type="email" class="form-control" id="contact-email" name="email" required
                                       value="<?php echo $currentUser ? htmlspecialchars($currentUser['email']) : ''; ?>">
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="contact-subject" class="form-label required"><?php echo $lang_data['subject']; ?></label>
                        <input type="text" class="form-control" id="contact-subject" name="subject" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="contact-message" class="form-label required"><?php echo $lang_data['message']; ?></label>
                        <textarea class="form-control" id="contact-message" name="message" rows="5" required></textarea>
                    </div>
                    
                    <div class="text-center">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-paper-plane me-2"></i><?php echo $lang_data['send_message']; ?>
                        </button>
                    </div>
                </form>
            </div>
            
            <div class="col-lg-6 animate-on-scroll">
                <div class="contact-info">
                    <h3 class="mb-4"><i class="fas fa-info-circle me-2"></i><?php echo $lang_data['get_in_touch']; ?></h3>
                    <p><?php echo $lang_data['contact_description']; ?></p>
                    
                    <div class="footer-contact">
                        <i class="fas fa-map-marker-alt"></i>
                        <div>
                            <h5><?php echo $lang_data['address']; ?></h5>
                            <p id="contact-address">123 Studio Street, Lomé, Togo</p>
                        </div>
                    </div>
                    
                    <div class="footer-contact">
                        <i class="fas fa-phone"></i>
                        <div>
                            <h5><?php echo $lang_data['phone']; ?></h5>
                            <p id="contact-phone">+228 12 345 6789</p>
                        </div>
                    </div>
                    
                    <div class="footer-contact">
                        <i class="fas fa-envelope"></i>
                        <div>
                            <h5><?php echo $lang_data['email']; ?></h5>
                            <p id="contact-email-text"><?php echo ADMIN_EMAIL; ?></p>
                        </div>
                    </div>
                    
                    <div class="social-icons mt-4">
                        <a href="#" aria-label="Facebook"><i class="fab fa-facebook-f"></i></a>
                        <a href="#" aria-label="Twitter"><i class="fab fa-twitter"></i></a>
                        <a href="#" aria-label="Instagram"><i class="fab fa-instagram"></i></a>
                        <a href="#" aria-label="YouTube"><i class="fab fa-youtube"></i></a>
                    </div>
                    
                    <div class="mt-4">
                        <iframe src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d252523.34709543096!2d1.0304166696883825!3d6.175602374453097!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x1023e1c113185419%3A0x2c3f1c80136b1c!2sLom%C3%A9%2C%20Togo!5e0!3m2!1sen!2sus!4v1650885966591!5m2!1sen!2sus" width="100%" height="250" style="border:0; border-radius: 10px;" allowfullscreen="" loading="lazy"></iframe>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Include custom JavaScript -->
<script src="assets/js/main.js"></script>

<?php
// Include footer
include 'includes/footer.php';
?>