<?php
/**
 * Payment Processing Class
 * Handles payment processing for bookings and pay-per-view content
 */

class Payment {
    private $db;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Process a booking payment
     * @param int $bookingId Booking ID
     * @param string $paymentMethod Payment method ('flooz', 'tmoney')
     * @param string $transactionId Transaction ID from the payment provider
     * @param float $amount Payment amount
     * @return array Result with success status and message
     */
    public function processBookingPayment($bookingId, $paymentMethod, $transactionId, $amount) {
        try {
            // Get booking information
            $query = "SELECT * FROM bookings WHERE id = ?";
            $booking = $this->db->getRow($query, [$bookingId]);
            
            if (!$booking) {
                return ['success' => false, 'message' => 'Booking not found'];
            }
            
            // Verify payment amount
            if ($amount < $booking['amount']) {
                return ['success' => false, 'message' => 'Payment amount is less than required'];
            }
            
            $this->db->beginTransaction();
            
            // Log the payment
            $paymentData = [
                'booking_id' => $bookingId,
                'amount' => $amount,
                'payment_method' => $paymentMethod,
                'transaction_id' => $transactionId,
                'status' => 'pending', // Initially pending until verified
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $paymentId = $this->db->insert('payments', $paymentData);
            
            // In a real application, you would verify the payment with the payment provider API
            // For this example, we'll simulate successful verification
            $verified = $this->verifyPayment($paymentMethod, $transactionId, $amount);
            
            if ($verified) {
                // Update payment status to success
                $this->db->update('payments', 
                    ['status' => 'completed', 'updated_at' => date('Y-m-d H:i:s')], 
                    'id = ?', 
                    [$paymentId]
                );
                
                // Update booking payment status
                $this->db->update('bookings', 
                    ['payment_status' => 'paid', 'updated_at' => date('Y-m-d H:i:s')], 
                    'id = ?', 
                    [$bookingId]
                );
                
                // If booking was pending, confirm it automatically
                if ($booking['status'] === 'pending') {
                    $this->db->update('bookings', 
                        ['status' => 'confirmed', 'updated_at' => date('Y-m-d H:i:s')], 
                        'id = ?', 
                        [$bookingId]
                    );
                }
                
                $this->db->commit();
                return ['success' => true, 'message' => 'Payment processed successfully'];
            } else {
                // Update payment status to failed
                $this->db->update('payments', 
                    ['status' => 'failed', 'updated_at' => date('Y-m-d H:i:s')], 
                    'id = ?', 
                    [$paymentId]
                );
                
                $this->db->commit();
                return ['success' => false, 'message' => 'Payment verification failed'];
            }
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollback();
            }
            error_log("Payment Error: " . $e->getMessage());
            return ['success' => false, 'message' => 'An error occurred while processing payment'];
        }
    }
    
    /**
     * Process a content purchase payment
     * @param int $userId User ID
     * @param int $contentId Content ID
     * @param string $paymentMethod Payment method ('flooz', 'tmoney')
     * @param string $transactionId Transaction ID from the payment provider
     * @param float $amount Payment amount
     * @return array Result with success status and message
     */
    public function processContentPayment($userId, $contentId, $paymentMethod, $transactionId, $amount) {
        try {
            // Get content information
            $query = "SELECT * FROM content WHERE id = ?";
            $content = $this->db->getRow($query, [$contentId]);
            
            if (!$content) {
                return ['success' => false, 'message' => 'Content not found'];
            }
            
            // Verify payment amount
            if ($amount < $content['price']) {
                return ['success' => false, 'message' => 'Payment amount is less than required'];
            }
            
            $this->db->beginTransaction();
            
            // Log the payment
            $paymentData = [
                'user_id' => $userId,
                'content_id' => $contentId,
                'amount' => $amount,
                'payment_method' => $paymentMethod,
                'transaction_id' => $transactionId,
                'status' => 'pending', // Initially pending until verified
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $paymentId = $this->db->insert('content_payments', $paymentData);
            
            // In a real application, you would verify the payment with the payment provider API
            // For this example, we'll simulate successful verification
            $verified = $this->verifyPayment($paymentMethod, $transactionId, $amount);
            
            if ($verified) {
                // Update payment status to success
                $this->db->update('content_payments', 
                    ['status' => 'completed', 'updated_at' => date('Y-m-d H:i:s')], 
                    'id = ?', 
                    [$paymentId]
                );
                
                // Grant user access to content
                $accessData = [
                    'user_id' => $userId,
                    'content_id' => $contentId,
                    'payment_id' => $paymentId,
                    'expires_at' => ($content['access_type'] === 'forever') ? null : date('Y-m-d H:i:s', strtotime('+30 days')),
                    'created_at' => date('Y-m-d H:i:s')
                ];
                
                $this->db->insert('user_content_access', $accessData);
                
                $this->db->commit();
                return ['success' => true, 'message' => 'Payment processed successfully'];
            } else {
                // Update payment status to failed
                $this->db->update('content_payments', 
                    ['status' => 'failed', 'updated_at' => date('Y-m-d H:i:s')], 
                    'id = ?', 
                    [$paymentId]
                );
                
                $this->db->commit();
                return ['success' => false, 'message' => 'Payment verification failed'];
            }
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollback();
            }
            error_log("Content Payment Error: " . $e->getMessage());
            return ['success' => false, 'message' => 'An error occurred while processing payment'];
        }
    }
    
    /**
     * Verify payment with payment provider
     * @param string $method Payment method
     * @param string $transactionId Transaction ID
     * @param float $amount Payment amount
     * @return bool True if payment is valid
     */
    private function verifyPayment($method, $transactionId, $amount) {
        // In a real application, this would make an API call to the payment provider
        // For this example, we'll simulate successful verification if transaction ID looks valid
        
        // Basic validation: Transaction ID should be at least 8 characters
        if (strlen($transactionId) < 8) {
            return false;
        }
        
        // Different validation based on payment method
        switch ($method) {
            case 'flooz':
                // Flooz transaction IDs typically start with FL
                return (stripos($transactionId, 'FL') === 0);
                
            case 'tmoney':
                // T-Money transaction IDs typically start with TM
                return (stripos($transactionId, 'TM') === 0);
                
            default:
                return false;
        }
    }
    
    /**
     * Check if user has access to content
     * @param int $userId User ID
     * @param int $contentId Content ID
     * @return bool True if user has access
     */
    public function hasContentAccess($userId, $contentId) {
        try {
            $query = "SELECT COUNT(*) FROM user_content_access 
                    WHERE user_id = ? AND content_id = ? 
                    AND (expires_at IS NULL OR expires_at > NOW())";
            
            return $this->db->getValue($query, [$userId, $contentId]) > 0;
        } catch (Exception $e) {
            error_log("Error checking content access: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get payment statistics
     * @param string $period Period to analyze ('day', 'week', 'month', 'year')
     * @return array Statistics
     */
    public function getPaymentStats($period = 'month') {
        try {
            // Initialize default statistics
            $stats = [
                'booking_revenue' => 0,
                'booking_count' => 0,
                'content_revenue' => 0,
                'content_count' => 0,
                'total_revenue' => 0,
                'by_method' => []
            ];
            
            // Set date condition based on period
            $dateCondition = '';
            
            switch ($period) {
                case 'day':
                    $dateCondition = "DATE(created_at) = CURDATE()";
                    break;
                case 'week':
                    $dateCondition = "YEARWEEK(created_at, 1) = YEARWEEK(CURDATE(), 1)";
                    break;
                case 'month':
                    $dateCondition = "YEAR(created_at) = YEAR(CURDATE()) AND MONTH(created_at) = MONTH(CURDATE())";
                    break;
                case 'year':
                    $dateCondition = "YEAR(created_at) = YEAR(CURDATE())";
                    break;
                default:
                    $dateCondition = "1=1"; // All time
            }
            
            // Safely check if tables exist
            if ($this->tableExists('payments')) {
                // Booking payments
                $bookingQuery = "SELECT 
                              SUM(amount) as total,
                              COUNT(*) as count
                             FROM payments 
                             WHERE status = 'completed' AND {$dateCondition}";
                $bookingStats = $this->db->getRow($bookingQuery);
                
                if ($bookingStats) {
                    $stats['booking_revenue'] = (float)($bookingStats['total'] ?? 0);
                    $stats['booking_count'] = (int)($bookingStats['count'] ?? 0);
                }
            }
            
            if ($this->tableExists('content_payments')) {
                // Content payments
                $contentQuery = "SELECT 
                              SUM(amount) as total,
                              COUNT(*) as count
                             FROM content_payments 
                             WHERE status = 'completed' AND {$dateCondition}";
                $contentStats = $this->db->getRow($contentQuery);
                
                if ($contentStats) {
                    $stats['content_revenue'] = (float)($contentStats['total'] ?? 0);
                    $stats['content_count'] = (int)($contentStats['count'] ?? 0);
                }
            }
            
            // Calculate total revenue
            $stats['total_revenue'] = $stats['booking_revenue'] + $stats['content_revenue'];
            
            // Only attempt to get payment methods if tables exist
            if ($this->tableExists('payments') && $this->tableExists('content_payments')) {
                try {
                    // Payment methods distribution - using a safer approach
                    $bookingMethods = $this->db->getRows("SELECT payment_method, COUNT(*) as count FROM payments WHERE status = 'completed' AND {$dateCondition} GROUP BY payment_method");
                    $contentMethods = $this->db->getRows("SELECT payment_method, COUNT(*) as count FROM content_payments WHERE status = 'completed' AND {$dateCondition} GROUP BY payment_method");
                    
                    // Combine payment methods
                    $methodCounts = [];
                    
                    // Process booking methods
                    foreach ($bookingMethods as $method) {
                        $methodName = $method['payment_method'];
                        $methodCounts[$methodName] = ($methodCounts[$methodName] ?? 0) + $method['count'];
                    }
                    
                    // Process content methods
                    foreach ($contentMethods as $method) {
                        $methodName = $method['payment_method'];
                        $methodCounts[$methodName] = ($methodCounts[$methodName] ?? 0) + $method['count'];
                    }
                    
                    // Convert to expected format
                    $byMethod = [];
                    foreach ($methodCounts as $name => $count) {
                        $byMethod[] = ['payment_method' => $name, 'count' => $count];
                    }
                    
                    // Sort by count descending
                    usort($byMethod, function($a, $b) {
                        return $b['count'] - $a['count'];
                    });
                    
                    $stats['by_method'] = $byMethod;
                } catch (Exception $e) {
                    error_log("Error getting payment methods: " . $e->getMessage());
                    // Keep default empty array for by_method
                }
            }
            
            return $stats;
        } catch (Exception $e) {
            error_log("Error getting payment stats: " . $e->getMessage());
            // Return default stats
            return [
                'booking_revenue' => 0,
                'booking_count' => 0,
                'content_revenue' => 0,
                'content_count' => 0,
                'total_revenue' => 0,
                'by_method' => []
            ];
        }
    }
    
    /**
     * Get recent payments
     * @param int $limit Number of payments to return
     * @return array Recent payments
     */
    public function getRecentPayments($limit = 10) {
        try {
            // Check if tables exist
            $paymentsExists = $this->tableExists('payments');
            $contentPaymentsExists = $this->tableExists('content_payments');
            
            if (!$paymentsExists && !$contentPaymentsExists) {
                // Neither payment table exists, return empty array
                return [];
            }
            
            $results = [];
            
            // If payments table exists, get booking payments
            if ($paymentsExists) {
                try {
                    // Safer query without JOINs that might fail
                    $bookingPayments = $this->db->getRows(
                        "SELECT p.*, 'booking' as payment_type FROM payments p 
                         WHERE p.status = 'completed' 
                         ORDER BY p.created_at DESC LIMIT ?",
                        [$limit]
                    );
                    
                    // Enhance with user and booking data if possible
                    foreach ($bookingPayments as &$payment) {
                        try {
                            if (isset($payment['booking_id'])) {
                                $booking = $this->db->getRow(
                                    "SELECT booking_reference, user_id FROM bookings WHERE id = ?", 
                                    [$payment['booking_id']]
                                );
                                
                                if ($booking) {
                                    $payment['booking_reference'] = $booking['booking_reference'];
                                    
                                    // Try to get user name
                                    if (isset($booking['user_id'])) {
                                        $user = $this->db->getRow(
                                            "SELECT name FROM users WHERE id = ?", 
                                            [$booking['user_id']]
                                        );
                                        
                                        if ($user) {
                                            $payment['user_name'] = $user['name'];
                                        }
                                    }
                                }
                            }
                        } catch (Exception $e) {
                            error_log("Error enhancing booking payment: " . $e->getMessage());
                            // Continue with basic payment data
                        }
                    }
                    
                    $results = array_merge($results, $bookingPayments);
                } catch (Exception $e) {
                    error_log("Error getting booking payments: " . $e->getMessage());
                    // Continue to try content payments
                }
            }
            
            // If content_payments table exists, get content payments
            if ($contentPaymentsExists) {
                try {
                    // Safer query without JOINs that might fail
                    $contentPayments = $this->db->getRows(
                        "SELECT cp.*, 'content' as payment_type FROM content_payments cp 
                         WHERE cp.status = 'completed' 
                         ORDER BY cp.created_at DESC LIMIT ?",
                        [$limit]
                    );
                    
                    // Enhance with user and content data if possible
                    foreach ($contentPayments as &$payment) {
                        try {
                            // Try to get user name
                            if (isset($payment['user_id'])) {
                                $user = $this->db->getRow(
                                    "SELECT name FROM users WHERE id = ?", 
                                    [$payment['user_id']]
                                );
                                
                                if ($user) {
                                    $payment['user_name'] = $user['name'];
                                }
                            }
                            
                            // Add content title if available
                            if (isset($payment['content_id'])) {
                                $content = $this->db->getRow(
                                    "SELECT title FROM content WHERE id = ?", 
                                    [$payment['content_id']]
                                );
                                
                                if ($content) {
                                    $payment['content_title'] = $content['title'];
                                }
                            }
                        } catch (Exception $e) {
                            error_log("Error enhancing content payment: " . $e->getMessage());
                            // Continue with basic payment data
                        }
                    }
                    
                    $results = array_merge($results, $contentPayments);
                } catch (Exception $e) {
                    error_log("Error getting content payments: " . $e->getMessage());
                    // Continue with any booking payments we have
                }
            }
            
            // Sort by created_at
            usort($results, function($a, $b) {
                return strtotime($b['created_at']) - strtotime($a['created_at']);
            });
            
            // Limit the final results
            return array_slice($results, 0, $limit);
            
        } catch (Exception $e) {
            error_log("Error in getRecentPayments: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Check if a table exists in the database
     * @param string $tableName Name of the table to check
     * @return bool True if table exists
     */
    private function tableExists($tableName) {
        try {
            $result = $this->db->getRow("SHOW TABLES LIKE ?", [$tableName]);
            return !empty($result);
        } catch (Exception $e) {
            error_log("Error checking if table exists: " . $e->getMessage());
            return false;
        }
    }
}