<?php
/**
 * Database Connection Class
 * Handles all database interactions for the application
 */

class Database {
    private $conn;
    private static $instance = null;
    
    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct() {
        try {
            $this->conn = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                ]
            );
        } catch (PDOException $e) {
            error_log("Database Connection Error: " . $e->getMessage());
            die("Database connection failed. Please try again later.");
        }
    }
    
    /**
     * Singleton pattern instance getter
     * @return Database The database instance
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        
        return self::$instance;
    }
    
    /**
     * Get database connection
     * @return PDO The PDO connection object
     */
    public function getConnection() {
        return $this->conn;
    }
    
    /**
     * Execute a query with parameters
     * @param string $query The SQL query
     * @param array $params The parameters for the query
     * @return PDOStatement The prepared statement after execution
     */
    public function query($query, $params = []) {
        try {
            $stmt = $this->conn->prepare($query);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            error_log("Query Error: " . $e->getMessage() . " - Query: " . $query);
            throw new Exception("Database query error");
        }
    }
    
    /**
     * Get a single row from the database
     * @param string $query The SQL query
     * @param array $params The parameters for the query
     * @return array|null The fetched row or null if no results
     */
    public function getRow($query, $params = []) {
        $stmt = $this->query($query, $params);
        return $stmt->fetch();
    }
    
    /**
     * Get multiple rows from the database
     * @param string $query The SQL query
     * @param array $params The parameters for the query
     * @return array The fetched rows
     */
    public function getRows($query, $params = []) {
        $stmt = $this->query($query, $params);
        return $stmt->fetchAll();
    }
    
    /**
     * Get a single value from the database
     * @param string $query The SQL query
     * @param array $params The parameters for the query
     * @return mixed The fetched value or null if no results
     */
    public function getValue($query, $params = []) {
        $stmt = $this->query($query, $params);
        $row = $stmt->fetch(PDO::FETCH_NUM);
        return $row ? $row[0] : null;
    }
    
    /**
     * Insert a row into the database
     * @param string $table The table name
     * @param array $data Associative array of column => value
     * @return int The last inserted ID
     */
    public function insert($table, $data) {
        $columns = implode(', ', array_keys($data));
        $placeholders = implode(', ', array_fill(0, count($data), '?'));
        
        $query = "INSERT INTO {$table} ({$columns}) VALUES ({$placeholders})";
        $this->query($query, array_values($data));
        
        return $this->conn->lastInsertId();
    }
    
    /**
     * Update rows in the database
     * @param string $table The table name
     * @param array $data Associative array of column => value to update
     * @param string $where The WHERE condition
     * @param array $params Parameters for the WHERE condition
     * @return int The number of affected rows
     */
    public function update($table, $data, $where, $params = []) {
        $setClauses = [];
        $updateParams = [];
        
        foreach ($data as $column => $value) {
            $setClauses[] = "{$column} = ?";
            $updateParams[] = $value;
        }
        
        $setClause = implode(', ', $setClauses);
        $query = "UPDATE {$table} SET {$setClause} WHERE {$where}";
        
        $stmt = $this->query($query, array_merge($updateParams, $params));
        return $stmt->rowCount();
    }
    
    /**
     * Delete rows from the database
     * @param string $table The table name
     * @param string $where The WHERE condition
     * @param array $params Parameters for the WHERE condition
     * @return int The number of affected rows
     */
    public function delete($table, $where, $params = []) {
        $query = "DELETE FROM {$table} WHERE {$where}";
        $stmt = $this->query($query, $params);
        return $stmt->rowCount();
    }
    
    /**
     * Begin a transaction
     */
    public function beginTransaction() {
        $this->conn->beginTransaction();
    }
    
    /**
     * Commit a transaction
     */
    public function commit() {
        $this->conn->commit();
    }
    
    /**
     * Rollback a transaction
     */
    public function rollback() {
        $this->conn->rollBack();
    }
}
