<?php
/**
 * Content Management Class
 * Handles all pay-per-view content functionality
 */

class Content {
    private $db;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Get all content
     * @param string $type Optional content type filter
     * @param string $status Optional status filter
     * @param int $limit Optional limit
     * @param int $offset Optional offset for pagination
     * @return array Content items
     */
    public function getAllContent($type = null, $status = 'published', $limit = null, $offset = null) {
        $query = "SELECT c.*, u.name as author_name, 
                  COUNT(DISTINCT v.id) as view_count,
                  COUNT(DISTINCT cp.id) as purchase_count
                  FROM content c 
                  LEFT JOIN users u ON c.author_id = u.id 
                  LEFT JOIN content_views v ON c.id = v.content_id
                  LEFT JOIN content_payments cp ON c.id = cp.content_id AND cp.status = 'completed'";
        
        $params = [];
        $conditions = [];
        
        if ($status) {
            $conditions[] = "c.status = ?";
            $params[] = $status;
        }
        
        if ($type) {
            $conditions[] = "c.content_type = ?";
            $params[] = $type;
        }
        
        if (!empty($conditions)) {
            $query .= " WHERE " . implode(" AND ", $conditions);
        }
        
        $query .= " GROUP BY c.id ORDER BY c.created_at DESC";
        
        if ($limit) {
            $query .= " LIMIT ?";
            $params[] = (int)$limit;
            
            if ($offset) {
                $query .= " OFFSET ?";
                $params[] = (int)$offset;
            }
        }
        
        return $this->db->getRows($query, $params);
    }
    
    /**
     * Get content by ID
     * @param int $id Content ID
     * @return array|null Content data or null if not found
     */
    public function getContentById($id) {
        $query = "SELECT c.*, u.name as author_name,
                  COUNT(DISTINCT v.id) as view_count,
                  COUNT(DISTINCT cp.id) as purchase_count
                  FROM content c 
                  LEFT JOIN users u ON c.author_id = u.id 
                  LEFT JOIN content_views v ON c.id = v.content_id
                  LEFT JOIN content_payments cp ON c.id = cp.content_id AND cp.status = 'completed'
                  WHERE c.id = ?
                  GROUP BY c.id";
        
        return $this->db->getRow($query, [$id]);
    }
    
    /**
     * Get featured content
     * @param int $limit Number of items to return
     * @return array Featured content
     */
    public function getFeaturedContent($limit = 6) {
        $query = "SELECT c.*, u.name as author_name,
                  COUNT(DISTINCT v.id) as view_count
                  FROM content c 
                  LEFT JOIN users u ON c.author_id = u.id 
                  LEFT JOIN content_views v ON c.id = v.content_id
                  WHERE c.status = 'published' AND c.is_featured = 1
                  GROUP BY c.id
                  ORDER BY c.created_at DESC
                  LIMIT ?";
        
        return $this->db->getRows($query, [$limit]);
    }
    
    /**
     * Get content by category
     * @param int $categoryId Category ID
     * @param int $limit Optional limit
     * @param int $offset Optional offset for pagination
     * @return array Content items
     */
    public function getContentByCategory($categoryId, $limit = null, $offset = null) {
        $query = "SELECT c.*, u.name as author_name,
                  COUNT(DISTINCT v.id) as view_count
                  FROM content c 
                  LEFT JOIN users u ON c.author_id = u.id 
                  LEFT JOIN content_views v ON c.id = v.content_id
                  WHERE c.status = 'published' AND c.category_id = ?
                  GROUP BY c.id
                  ORDER BY c.created_at DESC";
        
        $params = [$categoryId];
        
        if ($limit) {
            $query .= " LIMIT ?";
            $params[] = (int)$limit;
            
            if ($offset) {
                $query .= " OFFSET ?";
                $params[] = (int)$offset;
            }
        }
        
        return $this->db->getRows($query, $params);
    }
    
    /**
     * Search content
     * @param string $keyword Keyword to search for
     * @param int $limit Optional limit
     * @param int $offset Optional offset for pagination
     * @return array Content items
     */
    public function searchContent($keyword, $limit = null, $offset = null) {
        $searchTerm = "%{$keyword}%";
        
        $query = "SELECT c.*, u.name as author_name,
                  COUNT(DISTINCT v.id) as view_count
                  FROM content c 
                  LEFT JOIN users u ON c.author_id = u.id 
                  LEFT JOIN content_views v ON c.id = v.content_id
                  WHERE c.status = 'published' 
                  AND (c.title LIKE ? OR c.description LIKE ? OR c.tags LIKE ?)
                  GROUP BY c.id
                  ORDER BY c.created_at DESC";
        
        $params = [$searchTerm, $searchTerm, $searchTerm];
        
        if ($limit) {
            $query .= " LIMIT ?";
            $params[] = (int)$limit;
            
            if ($offset) {
                $query .= " OFFSET ?";
                $params[] = (int)$offset;
            }
        }
        
        return $this->db->getRows($query, $params);
    }
    
    /**
     * Get related content
     * @param int $contentId Current content ID
     * @param int $limit Number of items to return
     * @return array Related content
     */
    public function getRelatedContent($contentId, $limit = 4) {
        // Get current content details
        $content = $this->getContentById($contentId);
        
        if (!$content) {
            return [];
        }
        
        // Get content with same category or tags
        $query = "SELECT c.*, u.name as author_name
                  FROM content c 
                  LEFT JOIN users u ON c.author_id = u.id 
                  WHERE c.status = 'published' 
                  AND c.id != ? 
                  AND (c.category_id = ? OR c.tags LIKE ?)
                  ORDER BY c.created_at DESC
                  LIMIT ?";
        
        $tags = explode(',', $content['tags'] ?? '');
        $tagConditions = [];
        
        foreach ($tags as $tag) {
            $tag = trim($tag);
            if (!empty($tag)) {
                $tagConditions[] = "%{$tag}%";
            }
        }
        
        // If no tags, just use the category
        if (empty($tagConditions)) {
            $tagCondition = "%"; // Match anything
        } else {
            $tagCondition = $tagConditions[0]; // Use the first tag
        }
        
        return $this->db->getRows($query, [$contentId, $content['category_id'] ?? 0, $tagCondition, $limit]);
    }
    
    /**
     * Create new content
     * @param array $data Content data
     * @return int|bool The content ID if successful, false on failure
     */
    public function createContent($data) {
        try {
            // Generate slug from title
            if (!isset($data['slug']) || empty($data['slug'])) {
                $data['slug'] = $this->generateSlug($data['title']);
            }
            
            // Set default values
            if (!isset($data['status'])) {
                $data['status'] = 'draft';
            }
            
            if (!isset($data['is_featured'])) {
                $data['is_featured'] = 0;
            }
            
            if (!isset($data['created_at'])) {
                $data['created_at'] = date('Y-m-d H:i:s');
            }
            
            // Insert the content
            return $this->db->insert('content', $data);
        } catch (Exception $e) {
            error_log("Content Creation Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update content
     * @param int $id Content ID
     * @param array $data Updated content data
     * @return bool Success
     */
    public function updateContent($id, $data) {
        try {
            // Generate slug from title if title is being updated and slug isn't provided
            if (isset($data['title']) && (!isset($data['slug']) || empty($data['slug']))) {
                $data['slug'] = $this->generateSlug($data['title'], $id);
            }
            
            // Set updated timestamp
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            // Update the content
            return $this->db->update('content', $data, 'id = ?', [$id]) > 0;
        } catch (Exception $e) {
            error_log("Content Update Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete content
     * @param int $id Content ID
     * @return bool Success
     */
    public function deleteContent($id) {
        try {
            // In a production environment, consider soft delete instead
            return $this->db->delete('content', 'id = ?', [$id]) > 0;
        } catch (Exception $e) {
            error_log("Content Deletion Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Log content view
     * @param int $contentId Content ID
     * @param int $userId User ID (optional)
     * @return bool Success
     */
    public function logContentView($contentId, $userId = null) {
        try {
            $data = [
                'content_id' => $contentId,
                'user_id' => $userId,
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null,
                'viewed_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('content_views', $data);
            return true;
        } catch (Exception $e) {
            error_log("Content View Logging Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get user purchased content
     * @param int $userId User ID
     * @return array Purchased content
     */
    public function getUserPurchasedContent($userId) {
        $query = "SELECT c.*, uca.expires_at, cp.created_at as purchase_date
                  FROM user_content_access uca
                  JOIN content c ON uca.content_id = c.id
                  JOIN content_payments cp ON uca.payment_id = cp.id
                  WHERE uca.user_id = ? 
                  AND (uca.expires_at IS NULL OR uca.expires_at > NOW())
                  ORDER BY cp.created_at DESC";
        
        return $this->db->getRows($query, [$userId]);
    }
    
    /**
     * Get content categories
     * @return array Categories
     */
    public function getCategories() {
        $query = "SELECT c.*, COUNT(co.id) as content_count
                  FROM content_categories c
                  LEFT JOIN content co ON c.id = co.category_id
                  GROUP BY c.id
                  ORDER BY c.name ASC";
        
        return $this->db->getRows($query);
    }
    
    /**
     * Generate a URL-friendly slug from a string
     * @param string $string The string to convert
     * @param int $excludeId Optional ID to exclude when checking uniqueness
     * @return string The generated slug
     */
    private function generateSlug($string, $excludeId = null) {
        // Convert to lowercase
        $slug = strtolower($string);
        
        // Replace non-alphanumeric characters with hyphens
        $slug = preg_replace('/[^a-z0-9]+/', '-', $slug);
        
        // Remove hyphens from start and end
        $slug = trim($slug, '-');
        
        // Ensure the slug is unique
        $originalSlug = $slug;
        $count = 1;
        
        while (true) {
            $query = "SELECT COUNT(*) FROM content WHERE slug = ?";
            $params = [$slug];
            
            if ($excludeId) {
                $query .= " AND id != ?";
                $params[] = $excludeId;
            }
            
            $exists = $this->db->getValue($query, $params) > 0;
            
            if (!$exists) {
                break;
            }
            
            $slug = $originalSlug . '-' . $count;
            $count++;
        }
        
        return $slug;
    }
    
    /**
     * Get content statistics
     * @return array Statistics
     */
    public function getContentStats() {
        try {
            $stats = [
                'total' => 0,
                'total_views' => 0,
                'by_type' => [],
                'popular' => [],
                'recent' => [],
                'total_revenue' => 0
            ];
            
            // Total content
            $totalQuery = "SELECT COUNT(*) as total FROM content WHERE status = 'published'";
            $totalResult = $this->db->getRow($totalQuery);
            $stats['total'] = $totalResult ? (int)$totalResult['total'] : 0;
            
            // Total views
            $viewsQuery = "SELECT COUNT(*) as total_views FROM content_views";
            $viewsResult = $this->db->getRow($viewsQuery);
            $stats['total_views'] = $viewsResult ? (int)$viewsResult['total_views'] : 0;
            
            // Content by type
            $typeQuery = "SELECT content_type, COUNT(*) as count 
                          FROM content 
                          WHERE status = 'published'
                          GROUP BY content_type";
            $stats['by_type'] = $this->db->getRows($typeQuery) ?: [];
            
            // Popular content
            $popularQuery = "SELECT c.id, c.title, c.thumbnail, c.description, COUNT(v.id) as view_count
                            FROM content c
                            LEFT JOIN content_views v ON c.id = v.content_id
                            WHERE c.status = 'published'
                            GROUP BY c.id
                            ORDER BY view_count DESC
                            LIMIT 5";
            $stats['popular'] = $this->db->getRows($popularQuery) ?: [];
            
            // Recent content
            $recentQuery = "SELECT id, title, thumbnail, created_at
                           FROM content
                           WHERE status = 'published'
                           ORDER BY created_at DESC
                           LIMIT 5";
            $stats['recent'] = $this->db->getRows($recentQuery) ?: [];
            
            // Total revenue from content
            $revenueQuery = "SELECT SUM(amount) as total_revenue 
                            FROM content_payments 
                            WHERE status = 'completed'";
            $revenueResult = $this->db->getRow($revenueQuery);
            $stats['total_revenue'] = $revenueResult && $revenueResult['total_revenue'] ? (float)$revenueResult['total_revenue'] : 0;
            
            return $stats;
        } catch (Exception $e) {
            error_log("Error in getContentStats: " . $e->getMessage());
            // Return default stats in case of error
            return [
                'total' => 0,
                'total_views' => 0,
                'by_type' => [],
                'popular' => [],
                'recent' => [],
                'total_revenue' => 0
            ];
        }
    }
}