<?php
/**
 * Booking System Class
 * Handles all booking-related functionality
 */

class Booking {
    private $db;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Get all bookings
     * @param string $status Optional status filter
     * @param int $limit Optional limit
     * @param int $offset Optional offset for pagination
     * @return array Bookings
     */
    public function getAllBookings($status = null, $limit = null, $offset = null) {
        $query = "SELECT b.*, u.name as client_name, s.name as service_name 
                  FROM bookings b 
                  LEFT JOIN users u ON b.user_id = u.id 
                  LEFT JOIN services s ON b.service_id = s.id";
        
        $params = [];
        
        if ($status) {
            $query .= " WHERE b.status = ?";
            $params[] = $status;
        }
        
        $query .= " ORDER BY b.booking_date DESC, b.start_time ASC";
        
        if ($limit) {
            $query .= " LIMIT ?";
            $params[] = (int)$limit;
            
            if ($offset) {
                $query .= " OFFSET ?";
                $params[] = (int)$offset;
            }
        }
        
        return $this->db->getRows($query, $params);
    }
    
    /**
     * Get booking by ID
     * @param int $id Booking ID
     * @return array|null Booking data or null if not found
     */
    public function getBookingById($id) {
        $query = "SELECT b.*, u.name as client_name, u.email, u.phone, 
                  s.name as service_name, s.price
                  FROM bookings b 
                  LEFT JOIN users u ON b.user_id = u.id 
                  LEFT JOIN services s ON b.service_id = s.id
                  WHERE b.id = ?";
        
        return $this->db->getRow($query, [$id]);
    }
    
    /**
     * Get bookings for a specific user
     * @param int $userId User ID
     * @return array Bookings
     */
    public function getUserBookings($userId) {
        $query = "SELECT b.*, s.name as service_name, s.price 
                  FROM bookings b 
                  LEFT JOIN services s ON b.service_id = s.id
                  WHERE b.user_id = ? 
                  ORDER BY b.booking_date DESC, b.start_time ASC";
        
        return $this->db->getRows($query, [$userId]);
    }
    
    /**
     * Check if a time slot is available
     * @param string $date Booking date (YYYY-MM-DD)
     * @param string $startTime Start time (HH:MM:SS)
     * @param string $endTime End time (HH:MM:SS)
     * @param int $excludeId Optional booking ID to exclude (for updates)
     * @return bool True if available, false if already booked
     */
    public function isTimeSlotAvailable($date, $startTime, $endTime, $excludeId = null) {
        $query = "SELECT COUNT(*) FROM bookings 
                  WHERE booking_date = ? 
                  AND status IN ('pending', 'confirmed') 
                  AND (
                      (start_time <= ? AND end_time > ?) OR
                      (start_time < ? AND end_time >= ?) OR
                      (start_time >= ? AND end_time <= ?)
                  )";
        
        $params = [$date, $endTime, $startTime, $endTime, $startTime, $startTime, $endTime];
        
        if ($excludeId) {
            $query .= " AND id != ?";
            $params[] = $excludeId;
        }
        
        $count = $this->db->getValue($query, $params);
        return $count == 0;
    }
    
    /**
     * Get available time slots for a date
     * @param string $date Date to check (YYYY-MM-DD)
     * @return array Available time slots
     */
    public function getAvailableTimeSlots($date) {
        global $working_hours;
        
        // Get day of week (0 = Sunday, 1 = Monday, etc.)
        $dayOfWeek = date('w', strtotime($date));
        
        // If no working hours for this day, return empty array
        if (empty($working_hours[$dayOfWeek])) {
            return [];
        }
        
        // Get existing bookings for this date
        $query = "SELECT start_time, end_time FROM bookings 
                  WHERE booking_date = ? 
                  AND status IN ('pending', 'confirmed')
                  ORDER BY start_time ASC";
        
        $bookedSlots = $this->db->getRows($query, [$date]);
        
        // Create array of all possible time slots based on working hours
        $availableSlots = [];
        
        foreach ($working_hours[$dayOfWeek] as $hours) {
            list($startHour, $endHour) = explode('-', $hours);
            
            $startTime = strtotime($startHour . ':00');
            $endTime = strtotime($endHour . ':00');
            $slotLength = defined('TIMESLOT_LENGTH') ? TIMESLOT_LENGTH * 60 : 30 * 60; // Convert minutes to seconds, default to 30 min
            
            // Create slots at 30-minute intervals
            for ($time = $startTime; $time < $endTime; $time += 30 * 60) {
                $slotStart = date('H:i:00', $time);
                $slotEnd = date('H:i:00', $time + $slotLength);
                
                // Check if slot end is after working hours
                if (strtotime($slotEnd) > $endTime) {
                    continue;
                }
                
                // Check if this slot overlaps with any existing bookings
                $isAvailable = true;
                foreach ($bookedSlots as $bookedSlot) {
                    $bookedStart = $bookedSlot['start_time'];
                    $bookedEnd = $bookedSlot['end_time'];
                    
                    if (
                        ($slotStart <= $bookedEnd && $slotEnd > $bookedStart) ||
                        ($slotStart < $bookedEnd && $slotEnd >= $bookedStart) ||
                        ($slotStart >= $bookedStart && $slotEnd <= $bookedEnd)
                    ) {
                        $isAvailable = false;
                        break;
                    }
                }
                
                if ($isAvailable) {
                    // Format times for display (12-hour format)
                    $displayStart = date('g:i A', $time);
                    $displayEnd = date('g:i A', $time + $slotLength);
                    
                    $availableSlots[] = [
                        'value' => $slotStart . ' - ' . $slotEnd,
                        'display' => $displayStart . ' - ' . $displayEnd,
                        'start_time' => $slotStart,
                        'end_time' => $slotEnd
                    ];
                }
            }
        }
        
        return $availableSlots;
    }
    
    /**
     * Create a new booking
     * @param array $data Booking data
     * @return int|bool The booking ID if successful, false on failure
     */
    public function createBooking($data) {
        $this->db->beginTransaction();
        
        try {
            // Check if this is a guest booking or user booking
            $userId = isset($data['user_id']) ? $data['user_id'] : null;
            
            if (!$userId && isset($data['email'])) {
                // Check if user exists with this email
                $userQuery = "SELECT id FROM users WHERE email = ?";
                $userId = $this->db->getValue($userQuery, [$data['email']]);
                
                // Create user if not exists
                if (!$userId) {
                    $userData = [
                        'name' => $data['name'],
                        'email' => $data['email'],
                        'phone' => $data['phone'],
                        'role' => 'customer',
                        'created_at' => date('Y-m-d H:i:s')
                    ];
                    
                    $userId = $this->db->insert('users', $userData);
                }
            }
            
            // Parse time slot
            list($startTime, $endTime) = explode(' - ', $data['time_slot']);
            
            // Check if time slot is available
            if (!$this->isTimeSlotAvailable($data['date'], $startTime, $endTime)) {
                throw new Exception("The selected time slot is no longer available.");
            }
            
            // Generate booking reference
            $reference = $this->generateBookingReference();
            
            // Insert booking
            $bookingData = [
                'user_id' => $userId,
                'service_id' => $data['service_id'],
                'booking_reference' => $reference,
                'booking_date' => $data['date'],
                'start_time' => $startTime,
                'end_time' => $endTime,
                'notes' => $data['notes'] ?? null,
                'status' => 'pending',
                'payment_status' => 'unpaid',
                'amount' => $data['amount'],
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $bookingId = $this->db->insert('bookings', $bookingData);
            
            $this->db->commit();
            return $bookingId;
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Booking Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update booking status
     * @param int $id Booking ID
     * @param string $status New status
     * @return bool Success
     */
    public function updateBookingStatus($id, $status) {
        $validStatuses = ['pending', 'confirmed', 'completed', 'canceled'];
        
        if (!in_array($status, $validStatuses)) {
            return false;
        }
        
        return $this->db->update('bookings', 
            ['status' => $status, 'updated_at' => date('Y-m-d H:i:s')],
            'id = ?', 
            [$id]
        ) > 0;
    }
    
    /**
     * Update payment status
     * @param int $id Booking ID
     * @param string $status New payment status
     * @param string $transactionId Optional transaction ID
     * @return bool Success
     */
    public function updatePaymentStatus($id, $status, $transactionId = null) {
        $validStatuses = ['unpaid', 'partial', 'paid', 'refunded'];
        
        if (!in_array($status, $validStatuses)) {
            return false;
        }
        
        $data = [
            'payment_status' => $status,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        if ($transactionId) {
            $data['transaction_id'] = $transactionId;
        }
        
        return $this->db->update('bookings', $data, 'id = ?', [$id]) > 0;
    }
    
    /**
     * Get booking statistics
     * @param string $period Period to analyze ('day', 'week', 'month', 'year')
     * @return array Statistics
     */
    public function getBookingStats($period = 'month') {
        try {
            // Default stats to return in case of error
            $defaultStats = [
                'total' => 0,
                'pending' => 0,
                'confirmed' => 0,
                'completed' => 0, 
                'canceled' => 0,
                'total_revenue' => 0
            ];
            
            // Get date range based on period
            $dateRange = $this->getDateRangeForPeriod($period);
            $startDate = $dateRange['start'];
            $endDate = $dateRange['end'];
            
            // Total bookings
            $totalQuery = "SELECT COUNT(*) as total FROM bookings WHERE booking_date BETWEEN ? AND ?";
            $totalResult = $this->db->getRow($totalQuery, [$startDate, $endDate]);
            $total = $totalResult ? $totalResult['total'] : 0;
            
            // Bookings by status
            $byStatus = [
                'pending' => 0,
                'confirmed' => 0,
                'completed' => 0,
                'canceled' => 0
            ];
            
            foreach (array_keys($byStatus) as $status) {
                $statusQuery = "SELECT COUNT(*) as count FROM bookings WHERE status = ? AND booking_date BETWEEN ? AND ?";
                $statusResult = $this->db->getRow($statusQuery, [$status, $startDate, $endDate]);
                $byStatus[$status] = $statusResult ? $statusResult['count'] : 0;
            }
            
            // Total revenue
            $revenueQuery = "SELECT SUM(amount) as total_revenue FROM bookings WHERE payment_status = 'paid' AND booking_date BETWEEN ? AND ?";
            $revenueResult = $this->db->getRow($revenueQuery, [$startDate, $endDate]);
            $totalRevenue = $revenueResult && $revenueResult['total_revenue'] ? $revenueResult['total_revenue'] : 0;
            
            // Services distribution
            $servicesQuery = "SELECT s.name, COUNT(b.id) as count 
                             FROM bookings b
                             LEFT JOIN services s ON b.service_id = s.id
                             WHERE b.booking_date BETWEEN ? AND ?
                             GROUP BY s.name
                             ORDER BY count DESC";
            
            $services = $this->db->getRows($servicesQuery, [$startDate, $endDate]) ?: [];
            
            // Monthly data for chart
            $monthlyData = [];
            
            if ($period == 'year' || $period == 'all') {
                $monthlyQuery = "SELECT 
                                 MONTH(booking_date) as month,
                                 COUNT(*) as bookings,
                                 SUM(CASE WHEN payment_status = 'paid' THEN amount ELSE 0 END) as revenue
                                 FROM bookings
                                 WHERE booking_date BETWEEN ? AND ?
                                 GROUP BY MONTH(booking_date)
                                 ORDER BY MONTH(booking_date)";
                
                $monthlyData = $this->db->getRows($monthlyQuery, [$startDate, $endDate]) ?: [];
            }
            
            return [
                'total' => $total,
                'pending' => $byStatus['pending'],
                'confirmed' => $byStatus['confirmed'],
                'completed' => $byStatus['completed'],
                'canceled' => $byStatus['canceled'],
                'total_revenue' => $totalRevenue,
                'services' => $services,
                'monthly_data' => $monthlyData
            ];
        } catch (Exception $e) {
            error_log("Error in getBookingStats: " . $e->getMessage());
            // Return default stats in case of error
            return [
                'total' => 0,
                'pending' => 0,
                'confirmed' => 0,
                'completed' => 0,
                'canceled' => 0,
                'total_revenue' => 0,
                'services' => [],
                'monthly_data' => []
            ];
        }
    }
    
    /**
     * Get date range for the specified period
     * @param string $period The period (day, week, month, year, all)
     * @return array Start and end dates
     */
    private function getDateRangeForPeriod($period) {
        $today = date('Y-m-d');
        
        switch ($period) {
            case 'day':
                return [
                    'start' => $today,
                    'end' => $today
                ];
                
            case 'week':
                return [
                    'start' => date('Y-m-d', strtotime('monday this week')),
                    'end' => date('Y-m-d', strtotime('sunday this week'))
                ];
                
            case 'month':
                return [
                    'start' => date('Y-m-01'),
                    'end' => date('Y-m-t')
                ];
                
            case 'year':
                return [
                    'start' => date('Y-01-01'),
                    'end' => date('Y-12-31')
                ];
                
            case 'all':
            default:
                return [
                    'start' => '2000-01-01',
                    'end' => date('Y-12-31', strtotime('+10 years'))
                ];
        }
    }
    
    /**
     * Generate a unique booking reference
     * @return string Booking reference (e.g., WTV-2025-1234)
     */
    private function generateBookingReference() {
        $prefix = 'WTV';
        $year = date('Y');
        $randomNum = mt_rand(1000, 9999);
        
        $reference = "{$prefix}-{$year}-{$randomNum}";
        
        // Check if this reference already exists
        $query = "SELECT COUNT(*) FROM bookings WHERE booking_reference = ?";
        $exists = $this->db->getValue($query, [$reference]) > 0;
        
        // If it exists, try again with a different random number
        if ($exists) {
            return $this->generateBookingReference();
        }
        
        return $reference;
    }
    
    /**
     * Send booking confirmation SMS
     * @param int $bookingId The booking ID
     * @return bool Success
     */
    public function sendBookingConfirmationSMS($bookingId) {
        $booking = $this->getBookingById($bookingId);
        
        if (!$booking) {
            return false;
        }
        
        // Get user's phone number
        $phone = $booking['phone'];
        
        if (!$phone) {
            return false;
        }
        
        // Format date and time for SMS
        $date = date('d/m/Y', strtotime($booking['booking_date']));
        $time = date('g:i A', strtotime($booking['start_time'])) . ' - ' . 
                date('g:i A', strtotime($booking['end_time']));
        
        // Prepare message based on language preference
        // In a real app, you would detect the user's language preference
        $lang = 'en'; // Default to English
        
        if ($lang == 'en') {
            $message = "Your booking (Ref: {$booking['booking_reference']}) for {$booking['service_name']} on {$date} at {$time} has been confirmed. Thank you for choosing Web TV Studio.";
        } else {
            $message = "Votre réservation (Réf: {$booking['booking_reference']}) pour {$booking['service_name']} le {$date} à {$time} a été confirmée. Merci d'avoir choisi Web TV Studio.";
        }
        
        // In a real application, you would integrate with an SMS API
        // This is a placeholder for demonstration
        try {
            // Simulate SMS API call
            // sms_api_call($phone, $message);
            
            // Log the SMS
            $logData = [
                'booking_id' => $bookingId,
                'phone' => $phone,
                'message' => $message,
                'status' => 'sent',
                'sent_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('sms_logs', $logData);
            return true;
        } catch (Exception $e) {
            error_log("SMS Error: " . $e->getMessage());
            return false;
        }
    }
}