<?php
/**
 * Authentication and User Management Class
 * Handles user authentication, registration, and management
 * WITH ENHANCED ERROR PROTECTION
 */

class Auth {
    private $db;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Login user
     * @param string $email User email
     * @param string $password User password
     * @return array Result with user data or error message
     */
    public function login($email, $password) {
        try {
            // Validate inputs
            if (empty($email) || empty($password)) {
                return ['success' => false, 'message' => 'Email and password are required'];
            }
            
            // Get user by email
            $query = "SELECT * FROM users WHERE email = ?";
            $user = $this->db->getRow($query, [$email]);
            
            if (!$user) {
                return ['success' => false, 'message' => 'Invalid email or password'];
            }
            
            // Check if password field exists
            if (!isset($user['password'])) {
                error_log("User record missing password field for email: {$email}");
                return ['success' => false, 'message' => 'System error: Invalid user data'];
            }
            
            // Verify password
            if (!password_verify($password, $user['password'])) {
                return ['success' => false, 'message' => 'Invalid email or password'];
            }
            
            // Check if account is active
            if (!isset($user['status']) || $user['status'] !== 'active') {
                return ['success' => false, 'message' => 'Your account is not active'];
            }
            
            // Update last login
            $this->db->update('users', 
                ['last_login' => date('Y-m-d H:i:s')], 
                'id = ?', 
                [$user['id']]
            );
            
            // Remove password from user data
            unset($user['password']);
            
            // Ensure all expected fields exist
            $requiredFields = ['id', 'name', 'email', 'role'];
            foreach ($requiredFields as $field) {
                if (!isset($user[$field])) {
                    $user[$field] = null; // Set to null if missing
                }
            }
            
            // Store user in session
            $_SESSION['user'] = $user;
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['is_admin'] = (isset($user['role']) && $user['role'] === 'admin');
            
            return ['success' => true, 'user' => $user];
        } catch (Exception $e) {
            error_log("Login Error: " . $e->getMessage());
            return ['success' => false, 'message' => 'An error occurred during login'];
        }
    }
    
    /**
     * Register a new user
     * @param array $userData User data
     * @return array Result with user data or error message
     */
    public function register($userData) {
        try {
            // Validate required fields
            $requiredFields = ['name', 'email', 'password'];
            foreach ($requiredFields as $field) {
                if (!isset($userData[$field]) || empty($userData[$field])) {
                    return ['success' => false, 'message' => $field . ' is required'];
                }
            }
            
            // Check if email already exists
            $query = "SELECT COUNT(*) FROM users WHERE email = ?";
            $emailExists = $this->db->getValue($query, [$userData['email']]) > 0;
            
            if ($emailExists) {
                return ['success' => false, 'message' => 'Email already registered'];
            }
            
            // Hash password (ensure it exists)
            if (!isset($userData['password'])) {
                return ['success' => false, 'message' => 'Password is required'];
            }
            
            $userData['password'] = password_hash($userData['password'], PASSWORD_DEFAULT);
            
            // Set default values
            $userData['role'] = isset($userData['role']) ? $userData['role'] : 'customer';
            $userData['status'] = isset($userData['status']) ? $userData['status'] : 'active';
            $userData['created_at'] = date('Y-m-d H:i:s');
            
            // Insert user
            $userId = $this->db->insert('users', $userData);
            
            if ($userId) {
                // Get the user data without password
                $user = $this->getUserById($userId);
                
                if (!$user) {
                    return ['success' => false, 'message' => 'User was created but could not retrieve user data'];
                }
                
                // Make sure password is not set in the user data
                if (isset($user['password'])) {
                    unset($user['password']);
                }
                
                // Store user in session
                $_SESSION['user'] = $user;
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['is_admin'] = (isset($user['role']) && $user['role'] === 'admin');
                
                return ['success' => true, 'user' => $user];
            } else {
                return ['success' => false, 'message' => 'Registration failed'];
            }
        } catch (Exception $e) {
            error_log("Registration Error: " . $e->getMessage());
            return ['success' => false, 'message' => 'An error occurred during registration'];
        }
    }
    
    /**
     * Logout user
     * @return void
     */
    public function logout() {
        // Unset all session variables
        $_SESSION = [];
        
        // If it's desired to kill the session, also delete the session cookie
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }
        
        // Finally, destroy the session
        session_destroy();
    }
    
    /**
     * Check if user is logged in
     * @return bool True if logged in
     */
    public function isLoggedIn() {
        return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }
    
    /**
     * Check if logged in user is admin
     * @return bool True if admin
     */
    public function isAdmin() {
        return isset($_SESSION['is_admin']) && $_SESSION['is_admin'];
    }
    
    /**
     * Get current logged in user
     * @return array|null User data or null if not logged in
     */
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return isset($_SESSION['user']) ? $_SESSION['user'] : $this->getUserById($_SESSION['user_id']);
    }
    
    /**
     * Get user by ID
     * @param int $id User ID
     * @return array|null User data or null if not found
     */
    public function getUserById($id) {
        try {
            // Get all user data including password (will be removed before returning)
            $query = "SELECT * FROM users WHERE id = ?";
            $user = $this->db->getRow($query, [$id]);
            
            // If user is found, make sure it has a complete structure
            if ($user) {
                // Make sure all standard fields exist
                $standardFields = ['id', 'name', 'email', 'phone', 'role', 'status', 'created_at', 'last_login'];
                foreach ($standardFields as $field) {
                    if (!isset($user[$field])) {
                        $user[$field] = null;
                    }
                }
                
                // Add a null password field if it doesn't exist
                if (!isset($user['password'])) {
                    $user['password'] = null;
                }
            }
            
            return $user;
        } catch (Exception $e) {
            error_log("GetUserById Error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get all users
     * @param string $role Optional role filter
     * @param int $limit Optional limit
     * @param int $offset Optional offset for pagination
     * @return array Users
     */
    public function getAllUsers($role = null, $limit = null, $offset = null) {
        try {
            $query = "SELECT id, name, email, phone, role, status, created_at, last_login 
                      FROM users";
            
            $params = [];
            
            if ($role) {
                $query .= " WHERE role = ?";
                $params[] = $role;
            }
            
            $query .= " ORDER BY created_at DESC";
            
            if ($limit) {
                $query .= " LIMIT ?";
                $params[] = (int)$limit;
                
                if ($offset) {
                    $query .= " OFFSET ?";
                    $params[] = (int)$offset;
                }
            }
            
            return $this->db->getRows($query, $params);
        } catch (Exception $e) {
            error_log("GetAllUsers Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Update user
     * @param int $id User ID
     * @param array $data Updated user data
     * @return bool Success
     */
    public function updateUser($id, $data) {
        try {
            // If password is being updated, hash it
            if (isset($data['password']) && !empty($data['password'])) {
                $data['password'] = password_hash($data['password'], PASSWORD_DEFAULT);
            } else {
                // Don't update password if empty
                unset($data['password']);
            }
            
            // Update the user
            $success = $this->db->update('users', $data, 'id = ?', [$id]) > 0;
            
            // If the current user is being updated, update session
            if ($success && $this->isLoggedIn() && $_SESSION['user_id'] == $id) {
                $user = $this->getUserById($id);
                
                if ($user) {
                    // Remove password before storing in session
                    if (isset($user['password'])) {
                        unset($user['password']);
                    }
                    
                    $_SESSION['user'] = $user;
                    $_SESSION['is_admin'] = (isset($user['role']) && $user['role'] === 'admin');
                }
            }
            
            return $success;
        } catch (Exception $e) {
            error_log("User Update Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete user
     * @param int $id User ID
     * @return bool Success
     */
    public function deleteUser($id) {
        try {
            // Check if user exists
            $user = $this->getUserById($id);
            
            if (!$user) {
                return false;
            }
            
            // Don't allow deleting the current user
            if ($this->isLoggedIn() && $_SESSION['user_id'] == $id) {
                return false;
            }
            
            // In a production environment, consider soft delete instead
            return $this->db->delete('users', 'id = ?', [$id]) > 0;
        } catch (Exception $e) {
            error_log("User Deletion Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Reset password request
     * @param string $email User email
     * @return bool Success
     */
    public function requestPasswordReset($email) {
        try {
            // Check if email exists
            $query = "SELECT id FROM users WHERE email = ?";
            $userId = $this->db->getValue($query, [$email]);
            
            if (!$userId) {
                return false;
            }
            
            // Generate a unique token
            $token = bin2hex(random_bytes(32));
            $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
            
            // Delete any existing reset tokens for this user
            $this->db->delete('password_resets', 'user_id = ?', [$userId]);
            
            // Store the reset token
            $resetData = [
                'user_id' => $userId,
                'token' => $token,
                'expires_at' => $expiresAt,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('password_resets', $resetData);
            
            // Send reset email
            $resetUrl = defined('SITE_URL') ? SITE_URL . '/reset-password.php?token=' . $token : '/reset-password.php?token=' . $token;
            
            // In a real application, you would send an email with the reset link
            // This is a placeholder for demonstration
            // sendResetEmail($email, $resetUrl);
            
            return true;
        } catch (Exception $e) {
            error_log("Password Reset Request Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Reset password
     * @param string $token Reset token
     * @param string $password New password
     * @return bool Success
     */
    public function resetPassword($token, $password) {
        try {
            // Validate input
            if (empty($token) || empty($password)) {
                return false;
            }
            
            // Get reset token data
            $query = "SELECT * FROM password_resets WHERE token = ? AND expires_at > NOW()";
            $reset = $this->db->getRow($query, [$token]);
            
            if (!$reset || !isset($reset['user_id'])) {
                return false;
            }
            
            // Update user password
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            $success = $this->db->update('users', 
                ['password' => $hashedPassword], 
                'id = ?', 
                [$reset['user_id']]
            ) > 0;
            
            // Delete the used token
            if ($success) {
                $this->db->delete('password_resets', 'token = ?', [$token]);
            }
            
            return $success;
        } catch (Exception $e) {
            error_log("Password Reset Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get user statistics
     * @return array Statistics
     */
    public function getUserStats() {
        try {
            // Total users
            $totalQuery = "SELECT COUNT(*) FROM users";
            $total = $this->db->getValue($totalQuery);
            
            // Users by role
            $roleQuery = "SELECT role, COUNT(*) as count FROM users GROUP BY role";
            $byRole = $this->db->getRows($roleQuery);
            
            // New users this month
            $newUsersQuery = "SELECT COUNT(*) FROM users 
                             WHERE YEAR(created_at) = YEAR(CURDATE()) 
                             AND MONTH(created_at) = MONTH(CURDATE())";
            $newUsers = $this->db->getValue($newUsersQuery);
            
            // Users with bookings
            $activeQuery = "SELECT COUNT(DISTINCT user_id) FROM bookings";
            $activeUsers = $this->db->getValue($activeQuery);
            
            // Users with content purchases
            $payingUsersQuery = "SELECT COUNT(DISTINCT user_id) FROM content_payments WHERE status = 'completed'";
            $payingUsers = $this->db->getValue($payingUsersQuery);
            
            return [
                'total' => $total ?? 0,
                'by_role' => $byRole ?? [],
                'new_users' => $newUsers ?? 0,
                'active_users' => $activeUsers ?? 0,
                'paying_users' => $payingUsers ?? 0
            ];
        } catch (Exception $e) {
            error_log("Get User Stats Error: " . $e->getMessage());
            return [
                'total' => 0,
                'by_role' => [],
                'new_users' => 0,
                'active_users' => 0,
                'paying_users' => 0
            ];
        }
    }
    
    /**
     * Generate CSRF token
     * @return string CSRF token
     */
    public function generateCsrfToken() {
        if (!isset($_SESSION['csrf_token'])) {
            try {
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
            } catch (Exception $e) {
                // Fallback if random_bytes fails
                $_SESSION['csrf_token'] = md5(uniqid(mt_rand(), true));
            }
        }
        
        return $_SESSION['csrf_token'];
    }
    
    /**
     * Verify CSRF token
     * @param string $token Token to verify
     * @return bool Valid token
     */
    public function verifyCsrfToken($token) {
        if (!isset($_SESSION['csrf_token'])) {
            return false;
        }
        
        return hash_equals($_SESSION['csrf_token'], $token);
    }
}