<?php
/**
 * Content Payment Page
 * Handles payments for pay-per-view content
 */

// Include configuration
require_once 'includes/config.php';

// Include required classes
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/content.php';
require_once 'includes/payment.php';

// Initialize classes
$auth = new Auth();
$contentManager = new Content();
$paymentManager = new Payment();

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "languages/{$lang}.php";

// Check if user is logged in
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

// Get current user
$currentUser = $auth->getCurrentUser();
$userId = $currentUser['id'];

// Check if we have pending content payment
if (!isset($_SESSION['pending_content_payment']) || empty($_SESSION['pending_content_payment'])) {
    header('Location: index.php');
    exit;
}

$pendingPayment = $_SESSION['pending_content_payment'];
$contentId = $pendingPayment['content_id'];
$amount = $pendingPayment['amount'];
$paymentMethod = $pendingPayment['payment_method'];

// Get content details
$contentItem = $contentManager->getContentById($contentId);

if (!$contentItem) {
    // If content not found, clear session and redirect
    unset($_SESSION['pending_content_payment']);
    header('Location: index.php');
    exit;
}

// Check if user already has access
if ($paymentManager->hasContentAccess($userId, $contentId)) {
    // Clear session and redirect to watch page - already purchased
    unset($_SESSION['pending_content_payment']);
    header('Location: watch.php?id=' . $contentId);
    exit;
}

// Determine payment provider details
$paymentPhone = '';
$paymentInstructions = '';
$paymentLogo = '';

if ($paymentMethod === 'flooz') {
    $paymentPhone = FLOOZ_PHONE;
    $paymentLogo = 'assets/images/flooz.png';
    $paymentInstructions = $lang_data['flooz_instructions'];
} elseif ($paymentMethod === 'tmoney') {
    $paymentPhone = TMONEY_PHONE;
    $paymentLogo = 'assets/images/tmoney.png';
    $paymentInstructions = $lang_data['tmoney_instructions'];
}

// Handle payment form submission
$transactionError = '';
$paymentSuccess = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['transaction_id'])) {
    // Verify CSRF token
    if (!isset($_POST['csrf_token']) || !$auth->verifyCsrfToken($_POST['csrf_token'])) {
        $transactionError = $lang_data['invalid_token'];
    } else {
        $transactionId = trim($_POST['transaction_id']);
        
        // Basic validation: Transaction ID should be at least 8 characters
        if (strlen($transactionId) < 8) {
            $transactionError = $lang_data['invalid_transaction'];
        } else {
            // Process the content payment
            $result = $paymentManager->processContentPayment(
                $userId,
                $contentId,
                $paymentMethod,
                $transactionId,
                $amount
            );
            
            if ($result['success']) {
                // Payment successful
                $paymentSuccess = true;
                
                // Clear pending payment from session
                unset($_SESSION['pending_content_payment']);
            } else {
                $transactionError = $result['message'];
            }
        }
    }
}

// Get page title
$pageTitle = $lang_data['content_payment_title'] . ' - ' . SITE_NAME;

// Include header
include 'includes/header.php';
?>

<section class="payment-section py-5">
    <div class="container">
        <?php if ($paymentSuccess): ?>
            <!-- Payment Success -->
            <div class="card payment-success-card">
                <div class="card-body text-center">
                    <div class="mb-4">
                        <i class="fas fa-check-circle text-success" style="font-size: 5rem;"></i>
                    </div>
                    <h2 class="card-title mb-4"><?php echo $lang_data['payment_successful']; ?></h2>
                    <p><?php echo $lang_data['content_purchase_confirmed']; ?></p>
                    
                    <div class="content-details bg-light p-4 rounded my-4">
                        <div class="row">
                            <div class="col-md-6 text-md-end"><strong><?php echo $lang_data['content']; ?>:</strong></div>
                            <div class="col-md-6 text-md-start"><?php echo htmlspecialchars($contentItem['title']); ?></div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 text-md-end"><strong><?php echo $lang_data['amount']; ?>:</strong></div>
                            <div class="col-md-6 text-md-start"><?php echo number_format($amount) . ' ' . CURRENCY; ?></div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 text-md-end"><strong><?php echo $lang_data['access_type']; ?>:</strong></div>
                            <div class="col-md-6 text-md-start">
                                <?php 
                                if ($contentItem['access_type'] === 'forever') {
                                    echo $lang_data['lifetime_access'];
                                } else {
                                    echo $lang_data['access_period']; 
                                }
                                ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mt-4">
                        <a href="watch.php?id=<?php echo $contentId; ?>" class="btn btn-primary">
                            <?php echo $lang_data['watch_now']; ?>
                        </a>
                        <a href="my-content.php" class="btn btn-outline-primary">
                            <?php echo $lang_data['my_content']; ?>
                        </a>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <!-- Payment Form -->
            <div class="row justify-content-center">
                <div class="col-md-8">
                    <div class="card">
                        <div class="card-header bg-primary text-white">
                            <h3 class="card-title mb-0"><?php echo $lang_data['complete_content_payment']; ?></h3>
                        </div>
                        <div class="card-body">
                            <?php if (!empty($transactionError)): ?>
                                <div class="alert alert-danger">
                                    <?php echo $transactionError; ?>
                                </div>
                            <?php endif; ?>
                            
                            <div class="content-preview mb-4 text-center">
                                <img src="<?php echo htmlspecialchars($contentItem['thumbnail']); ?>" 
                                     alt="<?php echo htmlspecialchars($contentItem['title']); ?>" 
                                     class="img-fluid rounded mb-3" style="max-height: 200px;">
                                <h4><?php echo htmlspecialchars($contentItem['title']); ?></h4>
                            </div>
                            
                            <div class="payment-details mb-4">
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['content_type']; ?>:</div>
                                    <div class="col-md-8"><?php echo ucfirst($contentItem['content_type']); ?></div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['access_type']; ?>:</div>
                                    <div class="col-md-8">
                                        <?php 
                                        if ($contentItem['access_type'] === 'forever') {
                                            echo $lang_data['lifetime_access'];
                                        } else {
                                            echo $lang_data['access_period']; 
                                        }
                                        ?>
                                    </div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['amount']; ?>:</div>
                                    <div class="col-md-8 fw-bold text-primary">
                                        <?php echo number_format($amount) . ' ' . CURRENCY; ?>
                                    </div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-4 text-md-end fw-bold"><?php echo $lang_data['payment_method']; ?>:</div>
                                    <div class="col-md-8">
                                        <div class="d-flex align-items-center">
                                            <img src="<?php echo $paymentLogo; ?>" alt="<?php echo ucfirst($paymentMethod); ?>" 
                                                 style="height: 30px; margin-right: 10px;">
                                            <?php echo ucfirst($paymentMethod) . ' ' . $lang_data['mobile_money']; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="payment-instructions alert alert-info">
                                <h5><?php echo $lang_data['payment_instructions']; ?></h5>
                                <ol>
                                    <li><?php echo str_replace('{PHONE}', $paymentPhone, $paymentInstructions); ?></li>
                                    <li><?php echo $lang_data['enter_amount'] . ': ' . number_format($amount) . ' ' . CURRENCY; ?></li>
                                    <li><?php echo $lang_data['content_reference_info'] . ': ' . $contentItem['title']; ?></li>
                                    <li><?php echo $lang_data['confirm_payment']; ?></li>
                                    <li><?php echo $lang_data['enter_transaction_below']; ?></li>
                                </ol>
                                
                                <div class="text-center mt-3">
                                    <div class="payment-number">
                                        <?php echo $paymentPhone; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <form action="content-payment.php" method="post">
                                <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
                                
                                <div class="mb-3">
                                    <label for="transaction_id" class="form-label"><?php echo $lang_data['transaction_id']; ?></label>
                                    <input type="text" class="form-control" id="transaction_id" name="transaction_id" required>
                                    <div class="form-text"><?php echo $lang_data['transaction_help']; ?></div>
                                </div>
                                
                                <div class="d-grid gap-2">
                                    <button type="submit" class="btn btn-primary btn-lg">
                                        <?php echo $lang_data['confirm_payment_button']; ?>
                                    </button>
                                    <a href="watch.php?id=<?php echo $contentId; ?>" class="btn btn-outline-danger">
                                        <?php echo $lang_data['cancel_payment']; ?>
                                    </a>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</section>

<?php
// Include footer
include 'includes/footer.php';
?>
