<?php
/**
 * Book Studio Page
 * Allows users to book the studio for their productions
 */

// Include configuration
require_once 'includes/config.php';

// Include required classes
require_once 'includes/db.php';
require_once 'includes/auth.php';
require_once 'includes/booking.php';

// Initialize classes
$auth = new Auth();
$bookingManager = new Booking();
$db = Database::getInstance();

// Check if user is logged in
$isLoggedIn = $auth->isLoggedIn();
$currentUser = $isLoggedIn ? $auth->getCurrentUser() : null;

// Process booking submission
$bookingSuccess = false;
$bookingError = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_booking'])) {
    // Check if user is logged in
    if (!$isLoggedIn) {
        $bookingError = 'You must be logged in to book the studio.';
    } else {
        // Get booking data from form
        $serviceId = isset($_POST['service_id']) ? (int)$_POST['service_id'] : 0;
        $bookingDate = isset($_POST['booking_date']) ? $_POST['booking_date'] : '';
        $timeSlot = isset($_POST['time_slot']) ? $_POST['time_slot'] : '';
        $notes = isset($_POST['notes']) ? trim($_POST['notes']) : '';
        $amount = isset($_POST['amount']) ? (float)$_POST['amount'] : 0;
        $csrf_token = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';
        
        // Validate CSRF token
        if (empty($csrf_token) || !$auth->verifyCsrfToken($csrf_token)) {
            $bookingError = 'Invalid security token. Please try again.';
        }
        // Validate booking data
        elseif ($serviceId <= 0) {
            $bookingError = 'Please select a service.';
        }
        elseif (empty($bookingDate)) {
            $bookingError = 'Please select a booking date.';
        }
        elseif (empty($timeSlot)) {
            $bookingError = 'Please select a time slot.';
        }
        elseif ($amount <= 0) {
            $bookingError = 'Invalid booking amount.';
        }
        else {
            // Prepare booking data
            $bookingData = [
                'user_id' => $currentUser['id'],
                'service_id' => $serviceId,
                'date' => $bookingDate,
                'time_slot' => $timeSlot,
                'notes' => $notes,
                'amount' => $amount
            ];
            
            // Create booking
            $bookingId = $bookingManager->createBooking($bookingData);
            
            if ($bookingId) {
                // Redirect to payment page or show success message
                $bookingSuccess = true;
                
                if (defined('PAYMENT_ENABLED') && PAYMENT_ENABLED) {
                    // Redirect to payment page
                    header('Location: payment.php?booking_id=' . $bookingId);
                    exit;
                }
            } else {
                $bookingError = 'Failed to create booking. Please try again.';
            }
        }
    }
}

// Get available services
$services = $db->getRows("SELECT * FROM services WHERE status = 'active' ORDER BY name");

// Initialize date and time variables
$today = date('Y-m-d');
$maxDate = date('Y-m-d', strtotime('+3 months')); // Allow booking up to 3 months in advance

// Get selected date and service
$selectedDate = isset($_GET['date']) ? $_GET['date'] : '';
$selectedService = isset($_GET['service']) ? (int)$_GET['service'] : 0;

// Get available time slots if date is selected
$timeSlots = [];
if (!empty($selectedDate)) {
    $timeSlots = $bookingManager->getAvailableTimeSlots($selectedDate);
}

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "languages/{$lang}.php";

// Set page title
$pageTitle = $lang_data['book_studio'] . ' - ' . SITE_NAME;

// Include header
include 'includes/header.php';
?>

<!-- Book Studio Page -->
<section class="booking-hero">
    <div class="container">
        <div class="row">
            <div class="col-lg-12 text-center">
                <h1 class="display-4 fw-bold"><?php echo $lang_data['book_studio']; ?></h1>
                <p class="lead"><?php echo isset($lang_data['booking_subtitle']) ? $lang_data['booking_subtitle'] : 'Reserve our professional studio for your next production'; ?></p>
            </div>
        </div>
    </div>
</section>

<section class="booking-section py-5">
    <div class="container">
        <?php if ($bookingSuccess): ?>
            <div class="row justify-content-center">
                <div class="col-lg-8">
                    <div class="booking-success text-center py-5">
                        <div class="success-icon mb-4">
                            <i class="fas fa-check-circle fa-5x text-success"></i>
                        </div>
                        <h2><?php echo isset($lang_data['booking_success_title']) ? $lang_data['booking_success_title'] : 'Booking Successful!'; ?></h2>
                        <p class="lead"><?php echo $lang_data['booking_success']; ?></p>
                        <div class="mt-4">
                            <a href="profile.php#bookings-tab" class="btn btn-primary me-2">
                                <i class="fas fa-calendar-alt me-2"></i><?php echo $lang_data['view_my_bookings']; ?>
                            </a>
                            <a href="index.php" class="btn btn-outline-primary">
                                <i class="fas fa-home me-2"></i><?php echo $lang_data['back_to_home']; ?>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <div class="row">
                <!-- Booking Form -->
                <div class="col-lg-8">
                    <div class="card border-0 shadow-sm">
                        <div class="card-header bg-white py-3">
                            <h4 class="mb-0"><?php echo $lang_data['booking_form']; ?></h4>
                        </div>
                        <div class="card-body">
                            <?php if (!empty($bookingError)): ?>
                                <div class="alert alert-danger alert-dismissible fade show">
                                    <i class="fas fa-exclamation-circle me-2"></i> 
                                    <?php echo $bookingError; ?>
                                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (!$isLoggedIn): ?>
                                <div class="alert alert-info">
                                    <i class="fas fa-info-circle me-2"></i>
                                    <?php echo isset($lang_data['login_to_book']) ? $lang_data['login_to_book'] : 'Please log in to book our studio.'; ?>
                                    <a href="login.php?redirect=<?php echo urlencode($_SERVER['REQUEST_URI']); ?>" class="alert-link"><?php echo $lang_data['login']; ?></a>
                                    <?php echo isset($lang_data['or']) ? $lang_data['or'] : 'or'; ?>
                                    <a href="register.php" class="alert-link"><?php echo $lang_data['register']; ?></a>
                                </div>
                            <?php endif; ?>
                            
                            <form id="booking-form" action="book-studio.php" method="post">
                                <input type="hidden" name="csrf_token" value="<?php echo $isLoggedIn ? $auth->generateCsrfToken() : ''; ?>">
                                <input type="hidden" name="submit_booking" value="1">
                                
                                <div class="mb-4">
                                    <h5><?php echo $lang_data['select_service']; ?></h5>
                                    <div class="service-selection">
                                        <input type="hidden" name="service_id" id="service_id" value="<?php echo $selectedService; ?>">
                                        
                                        <div class="row">
                                            <?php foreach ($services as $service): ?>
                                                <div class="col-md-6 col-lg-4 mb-3">
                                                    <div class="service-option card h-100 <?php echo ($selectedService == $service['id']) ? 'active' : ''; ?>" 
                                                         data-id="<?php echo $service['id']; ?>"
                                                         data-price="<?php echo $service['price']; ?>"
                                                         data-duration="<?php echo $service['duration'] ?? 60; ?>">
                                                        <div class="card-body">
                                                            <div class="service-icon">
                                                                <i class="fas fa-<?php echo $service['icon'] ?? 'video'; ?>"></i>
                                                            </div>
                                                            <h5 class="service-name"><?php echo htmlspecialchars($service['name']); ?></h5>
                                                            <p class="service-description small"><?php echo htmlspecialchars($service['description']); ?></p>
                                                            <div class="service-price">
                                                                <span class="price"><?php echo number_format($service['price']); ?></span>
                                                                <span class="currency"><?php echo CURRENCY; ?></span>
                                                                <span class="price-unit">/ <?php echo isset($lang_data['hour']) ? $lang_data['hour'] : 'hour'; ?></span>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row mb-4">
                                    <div class="col-md-6">
                                        <h5><?php echo $lang_data['select_date']; ?></h5>
                                        <div class="input-group">
                                            <span class="input-group-text"><i class="fas fa-calendar"></i></span>
                                            <input type="date" class="form-control" id="booking_date" name="booking_date" 
                                                   min="<?php echo $today; ?>" max="<?php echo $maxDate; ?>" 
                                                   value="<?php echo $selectedDate; ?>" required>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="mb-4">
                                    <h5><?php echo $lang_data['select_time']; ?></h5>
                                    <div id="time-slots-container">
                                        <?php if (empty($selectedDate)): ?>
                                            <div class="alert alert-info">
                                                <i class="fas fa-info-circle me-2"></i>
                                                <?php echo isset($lang_data['select_date_first']) ? $lang_data['select_date_first'] : 'Please select a date to see available time slots.'; ?>
                                            </div>
                                        <?php elseif (empty($timeSlots)): ?>
                                            <div class="alert alert-warning">
                                                <i class="fas fa-exclamation-triangle me-2"></i>
                                                <?php echo $lang_data['no_slots_available']; ?>
                                            </div>
                                        <?php else: ?>
                                            <div class="time-slot-buttons">
                                                <?php foreach ($timeSlots as $slot): ?>
                                                    <button type="button" class="btn btn-outline-primary time-slot-btn m-1" 
                                                            data-value="<?php echo $slot['value']; ?>">
                                                        <?php echo $slot['display']; ?>
                                                    </button>
                                                <?php endforeach; ?>
                                            </div>
                                            <input type="hidden" name="time_slot" id="time_slot" required>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <div class="mb-4">
                                    <h5><?php echo $lang_data['additional_info']; ?></h5>
                                    <textarea class="form-control" id="notes" name="notes" rows="3" placeholder="<?php echo isset($lang_data['additional_info_placeholder']) ? $lang_data['additional_info_placeholder'] : 'Enter any additional requirements or special requests...'; ?>"></textarea>
                                </div>
                                
                                <input type="hidden" name="amount" id="total_amount" value="0">
                                
                                <div class="text-end">
                                    <button type="submit" class="btn btn-primary btn-lg" <?php echo $isLoggedIn ? '' : 'disabled'; ?>>
                                        <i class="fas fa-check-circle me-2"></i><?php echo $lang_data['submit_booking']; ?>
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
                
                <!-- Booking Summary -->
                <div class="col-lg-4">
                    <div class="card border-0 shadow-sm sticky-top" style="top: 20px;">
                        <div class="card-header bg-white py-3">
                            <h4 class="mb-0"><?php echo $lang_data['booking_summary']; ?></h4>
                        </div>
                        <div class="card-body">
                            <div class="summary-item mb-3">
                                <div class="d-flex justify-content-between">
                                    <span class="summary-label"><?php echo $lang_data['service']; ?></span>
                                    <span class="summary-value" id="selected_service">-</span>
                                </div>
                            </div>
                            
                            <div class="summary-item mb-3">
                                <div class="d-flex justify-content-between">
                                    <span class="summary-label"><?php echo $lang_data['date']; ?></span>
                                    <span class="summary-value" id="selected_date">-</span>
                                </div>
                            </div>
                            
                            <div class="summary-item mb-3">
                                <div class="d-flex justify-content-between">
                                    <span class="summary-label"><?php echo $lang_data['time']; ?></span>
                                    <span class="summary-value" id="selected_time">-</span>
                                </div>
                            </div>
                            
                            <div class="summary-item mb-3">
                                <div class="d-flex justify-content-between">
                                    <span class="summary-label"><?php echo $lang_data['duration']; ?></span>
                                    <span class="summary-value" id="booking_duration">0h 0m</span>
                                </div>
                            </div>
                            
                            <hr>
                            
                            <div class="summary-item mb-3">
                                <div class="d-flex justify-content-between align-items-center">
                                    <span class="summary-label fw-bold"><?php echo $lang_data['total']; ?></span>
                                    <span class="summary-value fs-4 fw-bold text-primary">
                                        <span id="booking_total">0</span>
                                        <span class="currency"><?php echo CURRENCY; ?></span>
                                    </span>
                                </div>
                            </div>
                            
                            <div class="booking-notes small mt-4">
                                <p class="text-muted mb-1">
                                    <i class="fas fa-info-circle me-1"></i>
                                    <?php echo isset($lang_data['booking_note_1']) ? $lang_data['booking_note_1'] : 'Bookings are confirmed upon payment.'; ?>
                                </p>
                                <p class="text-muted mb-1">
                                    <i class="fas fa-info-circle me-1"></i>
                                    <?php echo isset($lang_data['booking_note_2']) ? $lang_data['booking_note_2'] : 'Cancellations made 48 hours in advance are eligible for a full refund.'; ?>
                                </p>
                                <p class="text-muted mb-0">
                                    <i class="fas fa-info-circle me-1"></i>
                                    <?php echo isset($lang_data['booking_note_3']) ? $lang_data['booking_note_3'] : 'A staff member will contact you to confirm your booking details.'; ?>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</section>

<!-- Booking Page Styles -->
<style>
    /* Hero Section */
    .booking-hero {
        background-color: var(--bs-primary);
        color: white;
        padding: 60px 0;
        position: relative;
    }
    
    .booking-hero::after {
        content: '';
        position: absolute;
        bottom: 0;
        left: 0;
        right: 0;
        height: 50px;
        background-image: url('data:image/svg+xml;charset=utf8,%3Csvg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1440 48"%3E%3Cpath fill="%23ffffff" fill-opacity="1" d="M0,32L120,32C240,32,480,32,720,16C960,0,1200,0,1320,0L1440,0L1440,48L1320,48C1200,48,960,48,720,48C480,48,240,48,120,48L0,48Z"%3E%3C/path%3E%3C/svg%3E');
        background-size: cover;
        background-position: center;
    }
    
    /* Service Selection */
    .service-option {
        cursor: pointer;
        border: 2px solid #e0e0e0;
        transition: all 0.3s ease;
        border-radius: 8px;
    }
    
    .service-option:hover {
        border-color: var(--bs-primary);
        transform: translateY(-3px);
    }
    
    .service-option.active {
        border-color: var(--bs-primary);
        background-color: rgba(var(--bs-primary-rgb), 0.05);
    }
    
    .service-icon {
        font-size: 2rem;
        color: var(--bs-primary);
        margin-bottom: 15px;
    }
    
    .service-price {
        margin-top: 15px;
        font-weight: bold;
    }
    
    .service-price .price {
        font-size: 1.5rem;
        color: var(--bs-primary);
    }
    
    .service-price .price-unit {
        color: #6c757d;
        font-weight: normal;
    }
    
    /* Time Slots */
    .time-slot-btn {
        min-width: 120px;
    }
    
    .time-slot-btn.active {
        background-color: var(--bs-primary);
        color: white;
    }
    
    /* Booking Success */
    .booking-success {
        background-color: white;
        border-radius: 10px;
        box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    }
    
    .success-icon {
        color: var(--bs-success);
    }
</style>

<!-- Initialize Booking Form JavaScript -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Service selection
    const serviceOptions = document.querySelectorAll('.service-option');
    const serviceIdInput = document.getElementById('service_id');
    const selectedServiceDisplay = document.getElementById('selected_service');
    const totalAmountInput = document.getElementById('total_amount');
    const bookingTotalDisplay = document.getElementById('booking_total');
    
    serviceOptions.forEach(function(option) {
        option.addEventListener('click', function() {
            // Remove active class from all options
            serviceOptions.forEach(function(opt) {
                opt.classList.remove('active');
            });
            
            // Add active class to selected option
            this.classList.add('active');
            
            // Update hidden input
            const serviceId = this.dataset.id;
            serviceIdInput.value = serviceId;
            
            // Update summary display
            selectedServiceDisplay.textContent = this.querySelector('.service-name').textContent;
            
            // Calculate total
            calculateTotal();
        });
    });
    
    // Initialize service if preselected
    if (serviceIdInput.value) {
        const selectedOption = document.querySelector(`.service-option[data-id="${serviceIdInput.value}"]`);
        if (selectedOption) {
            selectedOption.classList.add('active');
            selectedServiceDisplay.textContent = selectedOption.querySelector('.service-name').textContent;
        }
    }
    
    // Date selection
    const dateInput = document.getElementById('booking_date');
    const selectedDateDisplay = document.getElementById('selected_date');
    
    dateInput.addEventListener('change', function() {
        // Update summary
        const date = new Date(this.value);
        const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
        selectedDateDisplay.textContent = date.toLocaleDateString(undefined, options);
        
        // Reload page with date parameter to get time slots
        window.location.href = 'book-studio.php?date=' + this.value + 
                              (serviceIdInput.value ? '&service=' + serviceIdInput.value : '');
    });
    
    // Initialize date display
    if (dateInput.value) {
        const date = new Date(dateInput.value);
        const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
        selectedDateDisplay.textContent = date.toLocaleDateString(undefined, options);
    }
    
    // Time slot selection
    const timeSlotButtons = document.querySelectorAll('.time-slot-btn');
    const timeSlotInput = document.getElementById('time_slot');
    const selectedTimeDisplay = document.getElementById('selected_time');
    const durationDisplay = document.getElementById('booking_duration');
    
    timeSlotButtons.forEach(function(button) {
        button.addEventListener('click', function() {
            // Remove active class from all buttons
            timeSlotButtons.forEach(function(btn) {
                btn.classList.remove('active');
            });
            
            // Add active class to clicked button
            this.classList.add('active');
            
            // Update hidden input
            timeSlotInput.value = this.dataset.value;
            
            // Update summary
            selectedTimeDisplay.textContent = this.textContent.trim();
            
            // Calculate duration and total
            calculateDuration();
            calculateTotal();
        });
    });
    
    // Calculate duration
    function calculateDuration() {
        if (!timeSlotInput.value) return;
        
        // Parse time slot (format: "HH:MM:SS - HH:MM:SS")
        const timeParts = timeSlotInput.value.split(' - ');
        const startTime = timeParts[0].split(':');
        const endTime = timeParts[1].split(':');
        
        // Calculate duration in minutes
        let startMinutes = parseInt(startTime[0]) * 60 + parseInt(startTime[1]);
        let endMinutes = parseInt(endTime[0]) * 60 + parseInt(endTime[1]);
        
        // Handle case when end time is on the next day
        if (endMinutes < startMinutes) {
            endMinutes += 24 * 60;
        }
        
        const durationMinutes = endMinutes - startMinutes;
        const hours = Math.floor(durationMinutes / 60);
        const minutes = durationMinutes % 60;
        
        // Update duration display
        durationDisplay.textContent = `${hours}h ${minutes}m`;
    }
    
    // Calculate total
    function calculateTotal() {
        if (!serviceIdInput.value || !timeSlotInput.value) {
            bookingTotalDisplay.textContent = '0';
            totalAmountInput.value = 0;
            return;
        }
        
        // Get service price
        const selectedService = document.querySelector(`.service-option[data-id="${serviceIdInput.value}"]`);
        const pricePerHour = parseFloat(selectedService.dataset.price);
        
        // Calculate duration
        const timeParts = timeSlotInput.value.split(' - ');
        const startTime = timeParts[0].split(':');
        const endTime = timeParts[1].split(':');
        
        let startMinutes = parseInt(startTime[0]) * 60 + parseInt(startTime[1]);
        let endMinutes = parseInt(endTime[0]) * 60 + parseInt(endTime[1]);
        
        if (endMinutes < startMinutes) {
            endMinutes += 24 * 60;
        }
        
        const durationHours = (endMinutes - startMinutes) / 60;
        
        // Calculate total
        const total = pricePerHour * durationHours;
        
        // Update displays
        bookingTotalDisplay.textContent = total.toFixed(0);
        totalAmountInput.value = total.toFixed(2);
    }
});
</script>

<?php
// Include footer
include 'includes/footer.php';
?>