/**
 * Main JavaScript for Web TV Studio
 * Controls main frontend functionality and UI interactions
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialize all components
    initNavbar();
    initScrollAnimations();
    initForms();
    initBackToTop();
    initVideoThumbnails();
    initTooltips();
    initLanguageSwitcher();
    initCarousels();
});

/**
 * Initialize navbar functionality
 */
function initNavbar() {
    // Add scroll event listener for navbar
    window.addEventListener('scroll', function() {
        const navbar = document.querySelector('.navbar');
        if (navbar) {
            if (window.scrollY > 50) {
                navbar.classList.add('scrolled');
            } else {
                navbar.classList.remove('scrolled');
            }
        }
    });
    
    // Trigger once to set initial state
    window.dispatchEvent(new Event('scroll'));
    
    // Handle navbar toggler for mobile
    const navbarToggler = document.querySelector('.navbar-toggler');
    if (navbarToggler) {
        navbarToggler.addEventListener('click', function() {
            document.body.classList.toggle('navbar-open');
        });
    }
    
    // Auto-close mobile menu when clicking outside
    document.addEventListener('click', function(event) {
        const navbar = document.querySelector('.navbar-collapse.show');
        if (navbar && !navbar.contains(event.target) && !event.target.classList.contains('navbar-toggler')) {
            const navbarToggler = document.querySelector('.navbar-toggler');
            if (navbarToggler && !navbarToggler.classList.contains('collapsed')) {
                navbarToggler.click();
            }
        }
    });
    
    // Handle dropdown menus on hover for desktop
    const dropdowns = document.querySelectorAll('.dropdown');
    if (window.innerWidth >= 992) {
        dropdowns.forEach(dropdown => {
            dropdown.addEventListener('mouseenter', function() {
                const dropdownMenu = this.querySelector('.dropdown-menu');
                if (dropdownMenu) {
                    dropdownMenu.classList.add('show');
                }
            });
            
            dropdown.addEventListener('mouseleave', function() {
                const dropdownMenu = this.querySelector('.dropdown-menu');
                if (dropdownMenu) {
                    dropdownMenu.classList.remove('show');
                }
            });
        });
    }
}

/**
 * Initialize scroll animations
 */
function initScrollAnimations() {
    // Get all elements with the animate-on-scroll class
    const animatedElements = document.querySelectorAll('.animate-on-scroll');
    
    if (animatedElements.length > 0) {
        // Check if IntersectionObserver is available
        if ('IntersectionObserver' in window) {
            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('animated');
                        observer.unobserve(entry.target);
                    }
                });
            }, {
                threshold: 0.1
            });
            
            animatedElements.forEach(element => {
                observer.observe(element);
            });
        } else {
            // Fallback for browsers that don't support IntersectionObserver
            animatedElements.forEach(element => {
                element.classList.add('animated');
            });
        }
    }
}

/**
 * Initialize form validations and interactions
 */
function initForms() {
    // Contact form validation
    const contactForm = document.getElementById('contact-form');
    if (contactForm) {
        contactForm.addEventListener('submit', function(e) {
            if (!validateForm(this)) {
                e.preventDefault();
            }
        });
    }
    
    // Newsletter form validation
    const newsletterForm = document.querySelector('.newsletter-form');
    if (newsletterForm) {
        newsletterForm.addEventListener('submit', function(e) {
            if (!validateForm(this)) {
                e.preventDefault();
            }
        });
    }
    
    // Form input enhancements
    const formControls = document.querySelectorAll('.form-control');
    formControls.forEach(control => {
        // Add focus class to parent when input is focused
        control.addEventListener('focus', function() {
            this.parentElement.classList.add('focused');
        });
        
        control.addEventListener('blur', function() {
            this.parentElement.classList.remove('focused');
            
            // Add filled class if input has value
            if (this.value.trim() !== '') {
                this.classList.add('is-filled');
            } else {
                this.classList.remove('is-filled');
            }
        });
        
        // Set initial state
        if (control.value.trim() !== '') {
            control.classList.add('is-filled');
        }
    });
}

/**
 * Validate form and show error messages
 * @param {HTMLFormElement} form - The form to validate
 * @returns {boolean} - True if form is valid, false otherwise
 */
function validateForm(form) {
    let isValid = true;
    const inputs = form.querySelectorAll('input, textarea, select');
    
    inputs.forEach(input => {
        // Reset validation state
        input.classList.remove('is-invalid');
        const errorMessage = input.nextElementSibling;
        if (errorMessage && errorMessage.classList.contains('invalid-feedback')) {
            errorMessage.remove();
        }
        
        // Check required fields
        if (input.hasAttribute('required') && input.value.trim() === '') {
            isValid = false;
            showError(input, 'This field is required');
        }
        
        // Check email format
        if (input.type === 'email' && input.value.trim() !== '') {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(input.value.trim())) {
                isValid = false;
                showError(input, 'Please enter a valid email address');
            }
        }
        
        // Check phone format
        if (input.type === 'tel' && input.value.trim() !== '') {
            const phoneRegex = /^\+?[0-9\s\-\(\)]{8,20}$/;
            if (!phoneRegex.test(input.value.trim())) {
                isValid = false;
                showError(input, 'Please enter a valid phone number');
            }
        }
        
        // Check password format
        if (input.type === 'password' && input.value.trim() !== '' && input.id === 'password') {
            if (input.value.length < 8) {
                isValid = false;
                showError(input, 'Password must be at least 8 characters long');
            }
        }
        
        // Check password confirmation
        if (input.type === 'password' && input.id === 'confirm_password') {
            const password = document.getElementById('password');
            if (password && input.value !== password.value) {
                isValid = false;
                showError(input, 'Passwords do not match');
            }
        }
    });
    
    return isValid;
}

/**
 * Show error message for form input
 * @param {HTMLElement} input - The input element
 * @param {string} message - The error message
 */
function showError(input, message) {
    input.classList.add('is-invalid');
    
    const errorDiv = document.createElement('div');
    errorDiv.className = 'invalid-feedback';
    errorDiv.textContent = message;
    
    // Insert error message after input
    input.parentNode.insertBefore(errorDiv, input.nextSibling);
}

/**
 * Initialize back to top button
 */
function initBackToTop() {
    const backToTopButton = document.querySelector('.back-to-top');
    
    if (backToTopButton) {
        // Show or hide button based on scroll position
        window.addEventListener('scroll', function() {
            if (window.scrollY > 300) {
                backToTopButton.classList.add('visible');
            } else {
                backToTopButton.classList.remove('visible');
            }
        });
        
        // Scroll to top when clicked
        backToTopButton.addEventListener('click', function() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        });
    }
}

/**
 * Initialize video thumbnails
 */
function initVideoThumbnails() {
    const videoThumbnails = document.querySelectorAll('.video-thumbnail');
    
    videoThumbnails.forEach(thumbnail => {
        thumbnail.addEventListener('click', function() {
            const videoId = this.getAttribute('data-video-id');
            const videoUrl = this.getAttribute('data-video-url');
            
            if (videoUrl) {
                window.location.href = videoUrl;
            } else if (videoId) {
                // This would typically open a modal with the video
                console.log('Video playback would start here for ID:', videoId);
            }
        });
    });
}

/**
 * Initialize tooltips
 */
function initTooltips() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    if (tooltipTriggerList.length > 0) {
        tooltipTriggerList.map(function(tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    }
}

/**
 * Initialize language switcher
 */
function initLanguageSwitcher() {
    const languageButtons = document.querySelectorAll('.language-switcher .btn');
    
    languageButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            
            const href = this.getAttribute('href');
            const lang = href.split('=')[1];
            
            // Set cookie for language preference
            document.cookie = `preferred_language=${lang}; path=/; max-age=31536000`; // 1 year
            
            // Redirect to the same page with the new language
            window.location.href = href;
        });
    });
}

/**
 * Initialize carousels
 */
function initCarousels() {
    const carousels = document.querySelectorAll('.carousel');
    
    if (carousels.length > 0) {
        carousels.forEach(carousel => {
            // Initialize Bootstrap carousel
            new bootstrap.Carousel(carousel, {
                interval: 5000,
                pause: 'hover'
            });
        });
    }
}

/**
 * Handle smooth scrolling for anchor links
 */
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function(e) {
        const targetId = this.getAttribute('href');
        
        if (targetId === '#') return;
        
        const targetElement = document.querySelector(targetId);
        
        if (targetElement) {
            e.preventDefault();
            
            // Close mobile menu if open
            const navbar = document.querySelector('.navbar-collapse.show');
            if (navbar) {
                const navbarToggler = document.querySelector('.navbar-toggler');
                if (navbarToggler) {
                    navbarToggler.click();
                }
            }
            
            // Calculate scroll position
            const headerHeight = document.querySelector('.navbar').offsetHeight;
            const targetPosition = targetElement.getBoundingClientRect().top + window.pageYOffset;
            const offsetPosition = targetPosition - headerHeight - 20;
            
            // Scroll to target
            window.scrollTo({
                top: offsetPosition,
                behavior: 'smooth'
            });
        }
    });
});

/**
 * Show flash messages and automatically dismiss them
 */
document.addEventListener('DOMContentLoaded', function() {
    // Flash messages
    const flashMessages = document.querySelectorAll('.alert-dismissible');
    
    flashMessages.forEach(message => {
        // Auto dismiss after 5 seconds
        setTimeout(() => {
            const closeButton = message.querySelector('.btn-close');
            if (closeButton) {
                closeButton.click();
            } else {
                message.classList.add('fade');
                setTimeout(() => {
                    message.remove();
                }, 500);
            }
        }, 5000);
    });
});