/**
 * Booking Form Functionality
 * Handles the booking form interactions, validation, and submission
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialize date and time pickers
    initDatepicker();
    initTimePicker();
    
    // Initialize service selection
    initServiceSelection();
    
    // Handle form validation and submission
    initBookingForm();
    
    // Initialize booking calendar if present
    initBookingCalendar();
});

/**
 * Initialize datepicker
 */
function initDatepicker() {
    const dateField = document.getElementById('booking_date');
    if (!dateField) return;
    
    // Get min and max date
    const today = new Date();
    const maxDate = new Date();
    maxDate.setMonth(maxDate.getMonth() + 3); // Allow booking up to 3 months in advance
    
    // Check if a flatpickr is available (it should be loaded in the page)
    if (typeof flatpickr !== 'undefined') {
        flatpickr(dateField, {
            minDate: "today",
            maxDate: maxDate,
            disable: [
                function(date) {
                    // Disable weekends if specified in the config
                    const disableWeekends = dateField.dataset.disableWeekends === 'true';
                    if (disableWeekends) {
                        return (date.getDay() === 0 || date.getDay() === 6);
                    }
                    return false;
                }
            ],
            locale: {
                firstDayOfWeek: 1 // Start from Monday
            },
            onChange: function(selectedDates, dateStr) {
                // Fetch available time slots for the selected date
                fetchAvailableTimeSlots(dateStr);
            }
        });
    } else {
        // Fallback to HTML5 date input
        const minDate = today.toISOString().split('T')[0];
        const maxDateStr = maxDate.toISOString().split('T')[0];
        
        dateField.setAttribute('min', minDate);
        dateField.setAttribute('max', maxDateStr);
        
        dateField.addEventListener('change', function() {
            fetchAvailableTimeSlots(this.value);
        });
    }
}

/**
 * Initialize time picker
 */
function initTimePicker() {
    const startTimeField = document.getElementById('start_time');
    const endTimeField = document.getElementById('end_time');
    
    if (!startTimeField || !endTimeField) return;
    
    // Check if a flatpickr is available
    if (typeof flatpickr !== 'undefined') {
        const timeConfig = {
            enableTime: true,
            noCalendar: true,
            dateFormat: "H:i",
            time_24hr: false,
            minuteIncrement: 30
        };
        
        flatpickr(startTimeField, {
            ...timeConfig,
            onChange: function(selectedDates, timeStr) {
                // Update end time to be at least 1 hour after start time
                if (selectedDates.length > 0) {
                    const minEndTime = new Date(selectedDates[0]);
                    minEndTime.setHours(minEndTime.getHours() + 1);
                    
                    if (endTimeField._flatpickr) {
                        endTimeField._flatpickr.set('minTime', minEndTime.getHours() + ":" + minEndTime.getMinutes());
                        
                        // If current end time is before min end time, update it
                        const currentEndTime = endTimeField._flatpickr.selectedDates[0];
                        if (currentEndTime && currentEndTime < minEndTime) {
                            endTimeField._flatpickr.setDate(minEndTime);
                        }
                    }
                }
                
                updateBookingDuration();
            }
        });
        
        flatpickr(endTimeField, {
            ...timeConfig
        });
        
        // Listen for changes to update duration
        endTimeField._flatpickr.config.onChange.push(function() {
            updateBookingDuration();
        });
    } else {
        // Fallback to HTML5 time input
        startTimeField.addEventListener('change', function() {
            // Ensure end time is after start time
            if (startTimeField.value && endTimeField.value) {
                const startTimeParts = startTimeField.value.split(':');
                const endTimeParts = endTimeField.value.split(':');
                
                const startTimeMinutes = parseInt(startTimeParts[0]) * 60 + parseInt(startTimeParts[1]);
                const endTimeMinutes = parseInt(endTimeParts[0]) * 60 + parseInt(endTimeParts[1]);
                
                if (endTimeMinutes <= startTimeMinutes + 60) {
                    // Set end time to 1 hour after start time
                    const newEndHour = Math.floor((startTimeMinutes + 60) / 60);
                    const newEndMinute = (startTimeMinutes + 60) % 60;
                    
                    endTimeField.value = 
                        (newEndHour < 10 ? '0' + newEndHour : newEndHour) + ':' + 
                        (newEndMinute < 10 ? '0' + newEndMinute : newEndMinute);
                }
            }
            
            updateBookingDuration();
        });
        
        endTimeField.addEventListener('change', function() {
            updateBookingDuration();
        });
    }
}

/**
 * Initialize service selection
 */
function initServiceSelection() {
    const serviceOptions = document.querySelectorAll('.service-option');
    
    serviceOptions.forEach(function(option) {
        option.addEventListener('click', function() {
            // Remove active class from all options
            serviceOptions.forEach(function(opt) {
                opt.classList.remove('active');
            });
            
            // Add active class to selected option
            this.classList.add('active');
            
            // Update hidden input
            const serviceIdInput = document.getElementById('service_id');
            if (serviceIdInput) {
                serviceIdInput.value = this.dataset.id;
            }
            
            // Update price display
            const priceDisplay = document.getElementById('selected_service_price');
            if (priceDisplay) {
                priceDisplay.textContent = this.dataset.price;
            }
            
            // Update duration display
            const durationDisplay = document.getElementById('selected_service_duration');
            if (durationDisplay) {
                durationDisplay.textContent = this.dataset.duration;
            }
            
            // Recalculate total
            calculateTotal();
        });
    });
}

/**
 * Initialize booking form
 */
function initBookingForm() {
    const bookingForm = document.getElementById('booking-form');
    if (!bookingForm) return;
    
    bookingForm.addEventListener('submit', function(e) {
        // Validate form
        if (!validateBookingForm()) {
            e.preventDefault();
            return false;
        }
        
        // Show loading state
        const submitButton = bookingForm.querySelector('button[type="submit"]');
        if (submitButton) {
            submitButton.disabled = true;
            submitButton.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';
        }
        
        // Form is valid, continue with submission
        return true;
    });
    
    // Add input listeners for real-time validation
    const inputs = bookingForm.querySelectorAll('input, select, textarea');
    inputs.forEach(function(input) {
        input.addEventListener('change', function() {
            validateField(this);
            calculateTotal();
        });
        
        input.addEventListener('blur', function() {
            validateField(this);
        });
    });
    
    // Initialize additional services checkboxes
    const additionalServices = document.querySelectorAll('.additional-service');
    additionalServices.forEach(function(service) {
        service.addEventListener('change', function() {
            calculateTotal();
        });
    });
}

/**
 * Initialize booking calendar
 */
function initBookingCalendar() {
    const calendarEl = document.getElementById('booking-calendar');
    if (!calendarEl) return;
    
    // Check if FullCalendar is available
    if (typeof FullCalendar !== 'undefined') {
        const calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth'
            },
            themeSystem: 'bootstrap5',
            selectable: true,
            selectMirror: true,
            unselectAuto: false,
            height: 'auto',
            contentHeight: 'auto',
            dateClick: function(info) {
                // Select the date in the form
                const dateField = document.getElementById('booking_date');
                if (dateField) {
                    dateField.value = info.dateStr;
                    
                    // Trigger change event
                    const event = new Event('change');
                    dateField.dispatchEvent(event);
                    
                    // If using flatpickr
                    if (dateField._flatpickr) {
                        dateField._flatpickr.setDate(info.dateStr);
                    }
                }
                
                // Scroll to booking form
                const bookingForm = document.getElementById('booking-form');
                if (bookingForm) {
                    bookingForm.scrollIntoView({ behavior: 'smooth' });
                }
            }
        });
        
        calendar.render();
    }
}

/**
 * Fetch available time slots for the selected date
 * @param {string} date Selected date in YYYY-MM-DD format
 */
function fetchAvailableTimeSlots(date) {
    const startTimeField = document.getElementById('start_time');
    const endTimeField = document.getElementById('end_time');
    
    if (!startTimeField || !endTimeField || !date) return;
    
    // Show loading indicator
    const timeSlotContainer = document.getElementById('time-slots-container');
    if (timeSlotContainer) {
        timeSlotContainer.innerHTML = '<div class="text-center py-3"><i class="fas fa-spinner fa-spin me-2"></i>Loading available time slots...</div>';
    }
    
    // In a real implementation, fetch time slots from server
    /*
    fetch('api/available-time-slots.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            date: date,
            service_id: document.getElementById('service_id')?.value || ''
        })
    })
    .then(response => response.json())
    .then(data => {
        updateTimeSlots(data.timeSlots);
    })
    .catch(error => {
        console.error('Error fetching time slots:', error);
        
        if (timeSlotContainer) {
            timeSlotContainer.innerHTML = '<div class="alert alert-danger">Failed to load available time slots. Please try again.</div>';
        }
    });
    */
    
    // For demo, use mock data
    setTimeout(function() {
        const timeSlots = [
            { start: '09:00', end: '10:00', available: true },
            { start: '10:00', end: '11:00', available: true },
            { start: '11:00', end: '12:00', available: true },
            { start: '12:00', end: '13:00', available: false },
            { start: '13:00', end: '14:00', available: true },
            { start: '14:00', end: '15:00', available: true },
            { start: '15:00', end: '16:00', available: false },
            { start: '16:00', end: '17:00', available: true }
        ];
        
        updateTimeSlots(timeSlots);
    }, 500);
}

/**
 * Update time slot options
 * @param {Array} timeSlots Available time slots
 */
function updateTimeSlots(timeSlots) {
    const timeSlotContainer = document.getElementById('time-slots-container');
    if (!timeSlotContainer) return;
    
    if (!timeSlots || timeSlots.length === 0) {
        timeSlotContainer.innerHTML = '<div class="alert alert-info">No time slots available for this date. Please select another date.</div>';
        return;
    }
    
    // Create time slot buttons
    let html = '<div class="time-slot-buttons">';
    
    timeSlots.forEach(function(slot) {
        const buttonClass = slot.available ? 'btn-outline-primary' : 'btn-outline-secondary disabled';
        html += `<button type="button" class="btn ${buttonClass} time-slot-btn m-1" 
                          data-start="${slot.start}" data-end="${slot.end}" 
                          ${!slot.available ? 'disabled' : ''}>
                    ${slot.start} - ${slot.end}
                </button>`;
    });
    
    html += '</div>';
    timeSlotContainer.innerHTML = html;
    
    // Add click event to time slot buttons
    const timeSlotButtons = timeSlotContainer.querySelectorAll('.time-slot-btn');
    timeSlotButtons.forEach(function(button) {
        button.addEventListener('click', function() {
            // Set start and end time
            const startTimeField = document.getElementById('start_time');
            const endTimeField = document.getElementById('end_time');
            
            if (startTimeField && endTimeField) {
                startTimeField.value = this.dataset.start;
                endTimeField.value = this.dataset.end;
                
                // If using flatpickr
                if (startTimeField._flatpickr) {
                    startTimeField._flatpickr.setDate(this.dataset.start);
                }
                
                if (endTimeField._flatpickr) {
                    endTimeField._flatpickr.setDate(this.dataset.end);
                }
                
                // Update booking duration
                updateBookingDuration();
            }
            
            // Update active state
            timeSlotButtons.forEach(function(btn) {
                btn.classList.remove('active');
            });
            
            this.classList.add('active');
        });
    });
}

/**
 * Update booking duration display
 */
function updateBookingDuration() {
    const startTimeField = document.getElementById('start_time');
    const endTimeField = document.getElementById('end_time');
    const durationDisplay = document.getElementById('booking_duration');
    
    if (!startTimeField || !endTimeField || !durationDisplay) return;
    
    if (startTimeField.value && endTimeField.value) {
        // Calculate duration in hours
        const startTimeParts = startTimeField.value.split(':');
        const endTimeParts = endTimeField.value.split(':');
        
        const startTimeMinutes = parseInt(startTimeParts[0]) * 60 + parseInt(startTimeParts[1]);
        const endTimeMinutes = parseInt(endTimeParts[0]) * 60 + parseInt(endTimeParts[1]);
        
        // If end time is before start time, assume it's next day
        let durationMinutes = endTimeMinutes - startTimeMinutes;
        if (durationMinutes < 0) {
            durationMinutes += 24 * 60;
        }
        
        const durationHours = Math.floor(durationMinutes / 60);
        const remainingMinutes = durationMinutes % 60;
        
        // Update duration display
        durationDisplay.textContent = `${durationHours}h ${remainingMinutes}m`;
        
        // Update hidden duration input
        const durationInput = document.getElementById('duration');
        if (durationInput) {
            durationInput.value = durationMinutes;
        }
        
        // Recalculate total
        calculateTotal();
    } else {
        durationDisplay.textContent = '0h 0m';
    }
}

/**
 * Calculate total booking amount
 */
function calculateTotal() {
    const serviceIdInput = document.getElementById('service_id');
    const durationInput = document.getElementById('duration');
    const totalDisplay = document.getElementById('booking_total');
    
    if (!serviceIdInput || !durationInput || !totalDisplay) return;
    
    if (serviceIdInput.value && durationInput.value) {
        // Get selected service
        const selectedService = document.querySelector(`.service-option[data-id="${serviceIdInput.value}"]`);
        if (!selectedService) return;
        
        // Get service price per hour
        const pricePerHour = parseFloat(selectedService.dataset.price);
        
        // Calculate base price based on duration
        const durationHours = parseInt(durationInput.value) / 60;
        let totalAmount = pricePerHour * durationHours;
        
        // Add additional services
        const additionalServices = document.querySelectorAll('.additional-service:checked');
        additionalServices.forEach(function(service) {
            totalAmount += parseFloat(service.dataset.price);
        });
        
        // Apply any discount if needed
        const discountInput = document.getElementById('discount');
        if (discountInput && discountInput.value) {
            const discount = parseFloat(discountInput.value);
            totalAmount = totalAmount - (totalAmount * (discount / 100));
        }
        
        // Update total display
        totalDisplay.textContent = totalAmount.toFixed(2);
        
        // Update hidden total input
        const totalInput = document.getElementById('total_amount');
        if (totalInput) {
            totalInput.value = totalAmount.toFixed(2);
        }
    } else {
        totalDisplay.textContent = '0.00';
    }
}

/**
 * Validate booking form
 * @returns {boolean} Form validity
 */
function validateBookingForm() {
    const bookingForm = document.getElementById('booking-form');
    if (!bookingForm) return false;
    
    const requiredFields = bookingForm.querySelectorAll('[required]');
    let isValid = true;
    
    // Validate each required field
    requiredFields.forEach(function(field) {
        if (!validateField(field)) {
            isValid = false;
        }
    });
    
    // Validate service selection
    const serviceIdInput = document.getElementById('service_id');
    if (serviceIdInput && !serviceIdInput.value) {
        showFieldError(serviceIdInput, 'Please select a service');
        isValid = false;
    }
    
    // Additional validation for booking date (must be in future)
    const dateField = document.getElementById('booking_date');
    if (dateField && dateField.value) {
        const selectedDate = new Date(dateField.value);
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        
        if (selectedDate < today) {
            showFieldError(dateField, 'Booking date must be in the future');
            isValid = false;
        }
    }
    
    // Additional validation for start and end time
    const startTimeField = document.getElementById('start_time');
    const endTimeField = document.getElementById('end_time');
    
    if (startTimeField && endTimeField && startTimeField.value && endTimeField.value) {
        // Ensure end time is after start time
        const startTimeParts = startTimeField.value.split(':');
        const endTimeParts = endTimeField.value.split(':');
        
        const startTimeMinutes = parseInt(startTimeParts[0]) * 60 + parseInt(startTimeParts[1]);
        const endTimeMinutes = parseInt(endTimeParts[0]) * 60 + parseInt(endTimeParts[1]);
        
        if (endTimeMinutes <= startTimeMinutes) {
            showFieldError(endTimeField, 'End time must be after start time');
            isValid = false;
        }
    }
    
    return isValid;
}

/**
 * Validate a single form field
 * @param {HTMLElement} field Field to validate
 * @returns {boolean} Field validity
 */
function validateField(field) {
    // Skip validation for non-required empty fields
    if (!field.hasAttribute('required') && !field.value) {
        clearFieldError(field);
        return true;
    }
    
    // Handle different input types
    switch (field.type) {
        case 'email':
            if (!field.value || !isValidEmail(field.value)) {
                showFieldError(field, 'Please enter a valid email address');
                return false;
            }
            break;
            
        case 'tel':
            if (!field.value || !isValidPhone(field.value)) {
                showFieldError(field, 'Please enter a valid phone number');
                return false;
            }
            break;
            
        case 'date':
            if (!field.value) {
                showFieldError(field, 'Please select a date');
                return false;
            }
            break;
            
        case 'time':
            if (!field.value) {
                showFieldError(field, 'Please select a time');
                return false;
            }
            break;
            
        default:
            if (!field.value.trim()) {
                showFieldError(field, 'This field is required');
                return false;
            }
    }
    
    // Field is valid
    clearFieldError(field);
    return true;
}

/**
 * Show error message for field
 * @param {HTMLElement} field Field with error
 * @param {string} message Error message
 */
function showFieldError(field, message) {
    // Clear any existing error
    clearFieldError(field);
    
    // Add error class to field
    field.classList.add('is-invalid');
    
    // Create error message element
    const errorElement = document.createElement('div');
    errorElement.className = 'invalid-feedback';
    errorElement.textContent = message;
    
    // Insert error message after field
    field.parentNode.appendChild(errorElement);
}

/**
 * Clear error message for field
 * @param {HTMLElement} field Field to clear error for
 */
function clearFieldError(field) {
    field.classList.remove('is-invalid');
    
    // Remove any existing error message
    const errorElement = field.parentNode.querySelector('.invalid-feedback');
    if (errorElement) {
        errorElement.remove();
    }
}

/**
 * Validate email format
 * @param {string} email Email to validate
 * @returns {boolean} Email validity
 */
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

/**
 * Validate phone number format
 * @param {string} phone Phone number to validate
 * @returns {boolean} Phone validity
 */
function isValidPhone(phone) {
    // Accept various phone formats
    const phoneRegex = /^[+]?[(]?[0-9]{3}[)]?[-\s.]?[0-9]{3}[-\s.]?[0-9]{4,6}$/;
    return phoneRegex.test(phone);
}