/**
 * Home Page Animations
 * Handles animations and interactive elements for the home page
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialize animations
    initScrollAnimations();
    initHeroAnimation();
    initCounterAnimations();
    initParallaxEffects();
    initTestimonialCarousel();
    
    // Initialize video player
    initVideoPlayers();
});

/**
 * Initialize scroll animations with Intersection Observer
 */
function initScrollAnimations() {
    // Animation elements
    const animatedElements = document.querySelectorAll('.fade-in, .fade-up, .fade-right, .fade-left, .scale-in');
    
    if (!animatedElements.length) return;
    
    // Create Intersection Observer
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animated');
                observer.unobserve(entry.target);
            }
        });
    }, {
        root: null,
        threshold: 0.1,
        rootMargin: '0px 0px -100px 0px'
    });
    
    // Observe each element
    animatedElements.forEach(element => {
        observer.observe(element);
    });
}

/**
 * Initialize hero section animation
 */
function initHeroAnimation() {
    const hero = document.querySelector('.hero-section');
    if (!hero) return;
    
    // Add animated class after a short delay
    setTimeout(() => {
        hero.classList.add('animated');
    }, 200);
    
    // Initialize typed.js if available
    const typedElement = document.querySelector('.typed-text');
    if (typedElement && window.Typed) {
        const strings = typedElement.dataset.strings?.split(',') || ['Welcome to our Studio', 'Create Amazing Content', 'Bring Your Ideas to Life'];
        
        new Typed(typedElement, {
            strings: strings,
            typeSpeed: 50,
            backSpeed: 30,
            backDelay: 2000,
            loop: true
        });
    }
}

/**
 * Initialize counter animations
 */
function initCounterAnimations() {
    const counters = document.querySelectorAll('.counter-number');
    if (!counters.length) return;
    
    // Create Intersection Observer for counters
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const target = entry.target;
                const targetValue = parseInt(target.getAttribute('data-target'));
                const duration = 2000; // 2 seconds
                let startTime = null;
                const startValue = 0;
                
                function updateCounter(timestamp) {
                    if (!startTime) startTime = timestamp;
                    const progress = timestamp - startTime;
                    const percentage = Math.min(progress / duration, 1);
                    
                    // Easing function (ease-out)
                    const easeOut = 1 - Math.pow(1 - percentage, 3);
                    
                    // Calculate current value
                    const currentValue = Math.floor(startValue + (targetValue - startValue) * easeOut);
                    
                    // Update counter
                    target.textContent = currentValue.toLocaleString();
                    
                    // Continue animation if not complete
                    if (percentage < 1) {
                        requestAnimationFrame(updateCounter);
                    }
                }
                
                requestAnimationFrame(updateCounter);
                observer.unobserve(target);
            }
        });
    }, {
        threshold: 0.5
    });
    
    // Observe each counter
    counters.forEach(counter => {
        observer.observe(counter);
    });
}

/**
 * Initialize parallax effects
 */
function initParallaxEffects() {
    const parallaxElements = document.querySelectorAll('.parallax-bg');
    if (!parallaxElements.length) return;
    
    // Update parallax position on scroll
    window.addEventListener('scroll', () => {
        const scrollTop = window.pageYOffset;
        
        parallaxElements.forEach(element => {
            const speed = element.dataset.speed || 0.5;
            const offset = -(scrollTop * speed);
            element.style.transform = `translateY(${offset}px)`;
        });
    });
}

/**
 * Initialize testimonial carousel
 */
function initTestimonialCarousel() {
    const carousel = document.querySelector('.testimonial-carousel');
    if (!carousel) return;
    
    // Check if Swiper is available
    if (typeof Swiper !== 'undefined') {
        new Swiper('.testimonial-carousel', {
            slidesPerView: 1,
            spaceBetween: 30,
            loop: true,
            autoplay: {
                delay: 5000,
                disableOnInteraction: false
            },
            pagination: {
                el: '.swiper-pagination',
                clickable: true
            },
            navigation: {
                nextEl: '.swiper-button-next',
                prevEl: '.swiper-button-prev'
            },
            breakpoints: {
                768: {
                    slidesPerView: 2
                },
                1024: {
                    slidesPerView: 3
                }
            }
        });
    } else {
        // Fallback to basic functionality if Swiper is not available
        const testimonials = carousel.querySelectorAll('.testimonial-item');
        if (testimonials.length <= 1) return;
        
        let currentIndex = 0;
        
        // Show specified testimonial
        function showTestimonial(index) {
            testimonials.forEach((testimonial, i) => {
                testimonial.style.display = i === index ? 'block' : 'none';
            });
            
            // Update navigation bullets if present
            const bullets = carousel.querySelectorAll('.carousel-bullet');
            bullets.forEach((bullet, i) => {
                bullet.classList.toggle('active', i === index);
            });
        }
        
        // Show first testimonial
        showTestimonial(currentIndex);
        
        // Add navigation buttons
        const prevButton = document.createElement('button');
        prevButton.className = 'carousel-prev';
        prevButton.innerHTML = '<i class="fas fa-chevron-left"></i>';
        prevButton.addEventListener('click', () => {
            currentIndex = (currentIndex - 1 + testimonials.length) % testimonials.length;
            showTestimonial(currentIndex);
        });
        
        const nextButton = document.createElement('button');
        nextButton.className = 'carousel-next';
        nextButton.innerHTML = '<i class="fas fa-chevron-right"></i>';
        nextButton.addEventListener('click', () => {
            currentIndex = (currentIndex + 1) % testimonials.length;
            showTestimonial(currentIndex);
        });
        
        // Create navigation dots
        const dotsContainer = document.createElement('div');
        dotsContainer.className = 'carousel-dots';
        
        testimonials.forEach((_, i) => {
            const dot = document.createElement('span');
            dot.className = 'carousel-bullet' + (i === 0 ? ' active' : '');
            dot.addEventListener('click', () => {
                currentIndex = i;
                showTestimonial(currentIndex);
            });
            dotsContainer.appendChild(dot);
        });
        
        // Add navigation to carousel
        const navigation = document.createElement('div');
        navigation.className = 'carousel-navigation';
        navigation.appendChild(prevButton);
        navigation.appendChild(dotsContainer);
        navigation.appendChild(nextButton);
        
        carousel.appendChild(navigation);
        
        // Auto-rotate testimonials
        setInterval(() => {
            currentIndex = (currentIndex + 1) % testimonials.length;
            showTestimonial(currentIndex);
        }, 5000);
    }
}

/**
 * Initialize video players
 */
function initVideoPlayers() {
    const videoTriggers = document.querySelectorAll('.video-trigger');
    
    videoTriggers.forEach(trigger => {
        trigger.addEventListener('click', function(e) {
            e.preventDefault();
            
            const videoId = this.dataset.videoId;
            const videoType = this.dataset.videoType || 'youtube';
            
            if (!videoId) return;
            
            // Create modal
            const modal = document.createElement('div');
            modal.className = 'video-modal';
            modal.innerHTML = `
                <div class="video-modal-overlay"></div>
                <div class="video-modal-container">
                    <button class="video-modal-close">&times;</button>
                    <div class="video-modal-content">
                        ${getVideoEmbed(videoId, videoType)}
                    </div>
                </div>
            `;
            
            document.body.appendChild(modal);
            
            // Prevent body scrolling
            document.body.style.overflow = 'hidden';
            
            // Add animation
            setTimeout(() => {
                modal.classList.add('active');
            }, 10);
            
            // Close modal on click
            modal.querySelector('.video-modal-overlay').addEventListener('click', closeModal);
            modal.querySelector('.video-modal-close').addEventListener('click', closeModal);
            
            function closeModal() {
                modal.classList.remove('active');
                
                // Remove modal after animation
                setTimeout(() => {
                    document.body.removeChild(modal);
                    document.body.style.overflow = '';
                }, 300);
            }
        });
    });
}

/**
 * Get video embed code
 * @param {string} videoId Video ID
 * @param {string} videoType Video type (youtube, vimeo)
 * @returns {string} Embed HTML
 */
function getVideoEmbed(videoId, videoType) {
    switch (videoType.toLowerCase()) {
        case 'youtube':
            return `<iframe width="100%" height="100%" src="https://www.youtube.com/embed/${videoId}?autoplay=1" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe>`;
            
        case 'vimeo':
            return `<iframe width="100%" height="100%" src="https://player.vimeo.com/video/${videoId}?autoplay=1" frameborder="0" allow="autoplay; fullscreen" allowfullscreen></iframe>`;
            
        case 'direct':
            return `<video width="100%" height="100%" controls autoplay><source src="${videoId}" type="video/mp4">Your browser does not support the video tag.</video>`;
            
        default:
            return `<div class="error-message">Unsupported video type</div>`;
    }
}