<?php
/**
 * Admin Login Page
 * Authentication for admin panel access
 */

// For debugging if needed
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include configuration
require_once '../includes/config.php';

// Include required classes
require_once '../includes/db.php';
require_once '../includes/auth.php';

// Initialize classes
$auth = new Auth();

// Check if user is already logged in and is admin
if ($auth->isLoggedIn() && $auth->isAdmin()) {
    // Redirect to admin dashboard
    header('Location: index.php');
    exit;
}

// Process login form
$loginError = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Safely get POST variables
    $email = isset($_POST['email']) ? trim($_POST['email']) : '';
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    $rememberMe = isset($_POST['remember_me']) && $_POST['remember_me'] == '1';
    $csrf_token = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';

    // Verify CSRF token
    if (empty($csrf_token) || !$auth->verifyCsrfToken($csrf_token)) {
        $loginError = 'Invalid security token. Please try again.';
    } else {
        // Validate email and password
        if (empty($email) || empty($password)) {
            $loginError = 'Please enter both email and password.';
        } else {
            // Attempt login
            $result = $auth->login($email, $password);
            
            if ($result['success']) {
                // Check if user is an admin
                if ($auth->isAdmin()) {
                    // Set remember me cookie if requested
                    if ($rememberMe) {
                        // Generate token and store in database
                        $token = bin2hex(random_bytes(32));
                        $userId = $result['user']['id'];
                        $expiresAt = date('Y-m-d H:i:s', strtotime('+30 days'));
                        
                        $db = Database::getInstance();
                        
                        // Check if remember_tokens table exists
                        try {
                            $tableExists = $db->getValue("SHOW TABLES LIKE 'remember_tokens'");
                            
                            if ($tableExists) {
                                $db->query(
                                    "INSERT INTO remember_tokens (user_id, token, expires_at, created_at) 
                                    VALUES (?, ?, ?, NOW())",
                                    [$userId, $token, $expiresAt]
                                );
                                
                                // Set cookie
                                setcookie('remember_token', $token, time() + (86400 * 30), '/', '', true, true);
                            }
                        } catch (Exception $e) {
                            // Silently fail if remember tokens don't work
                            error_log("Remember token error: " . $e->getMessage());
                        }
                    }
                    
                    // Redirect to admin dashboard
                    header('Location: index.php');
                    exit;
                } else {
                    // Not an admin user
                    $auth->logout();
                    $loginError = 'You do not have permission to access the admin area.';
                }
            } else {
                $loginError = $result['message'];
            }
        }
    }
}

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "../languages/{$lang}.php";

// Set page title
$pageTitle = 'Admin Login - ' . SITE_NAME;
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?></title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    
    <!-- Admin CSS -->
    <link rel="stylesheet" href="../assets/admin/css/dashboard.css">
    
    <style>
        body {
            background-color: #f8f9fa;
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .login-card {
            max-width: 400px;
            width: 100%;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            background-color: white;
        }
        .login-logo {
            text-align: center;
            margin-bottom: 30px;
        }
        .login-logo img {
            max-height: 70px;
        }
    </style>
</head>
<body>
    <div class="login-card">
        <div class="login-logo">
            <img src="../assets/images/logo.png" alt="<?php echo SITE_NAME; ?>">
            <h4 class="mt-3 text-primary">Admin Login</h4>
        </div>
        
        <?php if (!empty($loginError)): ?>
            <div class="alert alert-danger">
                <?php echo $loginError; ?>
            </div>
        <?php endif; ?>
        
        <form method="post" action="login.php">
            <input type="hidden" name="csrf_token" value="<?php echo $auth->generateCsrfToken(); ?>">
            
            <div class="mb-3">
                <label for="email" class="form-label"><?php echo $lang_data['email_address']; ?></label>
                <div class="input-group">
                    <span class="input-group-text"><i class="fas fa-envelope"></i></span>
                    <input type="email" class="form-control" id="email" name="email" required 
                           value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>">
                </div>
            </div>
            
            <div class="mb-3">
                <label for="password" class="form-label"><?php echo $lang_data['password']; ?></label>
                <div class="input-group">
                    <span class="input-group-text"><i class="fas fa-lock"></i></span>
                    <input type="password" class="form-control" id="password" name="password" required>
                </div>
            </div>
            
            <div class="mb-3 form-check">
                <input type="checkbox" class="form-check-input" id="remember_me" name="remember_me" value="1">
                <label class="form-check-label" for="remember_me"><?php echo $lang_data['remember_me']; ?></label>
            </div>
            
            <div class="d-grid">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-sign-in-alt me-2"></i><?php echo $lang_data['login']; ?>
                </button>
            </div>
        </form>
        
        <div class="text-center mt-3">
            <a href="../index.php" class="text-decoration-none">
                <i class="fas fa-arrow-left me-1"></i> Back to Website
            </a>
        </div>
    </div>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>