/**
 * Admin Dashboard JavaScript
 * Handles all dashboard functionality including charts, calendar, and data updates
 */

// Wait for DOM to be ready
document.addEventListener('DOMContentLoaded', function() {
    // Initialize dashboard components
    initSidebar();
    initStats();
    initCharts();
    initCalendar();
    initDataTables();
    initEventListeners();
    
    // Show success message if present in URL
    showMessagesFromUrl();
});

/**
 * Initialize sidebar functionality
 */
function initSidebar() {
    // Toggle sidebar on mobile
    const sidebarToggler = document.getElementById('sidebarToggler');
    if (sidebarToggler) {
        sidebarToggler.addEventListener('click', function() {
            document.body.classList.toggle('sidebar-collapsed');
        });
    }
    
    // Add active class to current page link
    const currentPage = window.location.pathname.split('/').pop();
    const sidebarLinks = document.querySelectorAll('.sidebar-nav .nav-link');
    
    sidebarLinks.forEach(link => {
        const href = link.getAttribute('href');
        if (href === currentPage || (currentPage === '' && href === 'index.php')) {
            link.classList.add('active');
        }
    });
    
    // Collapse/expand sidebar menus
    const dropdownToggles = document.querySelectorAll('.sidebar-dropdown-toggle');
    
    dropdownToggles.forEach(toggle => {
        toggle.addEventListener('click', function(e) {
            e.preventDefault();
            const parent = this.parentElement;
            parent.classList.toggle('show');
            
            // Find the dropdown menu
            const dropdownMenu = parent.querySelector('.sidebar-dropdown-menu');
            
            // Toggle the dropdown menu
            if (parent.classList.contains('show')) {
                dropdownMenu.style.maxHeight = dropdownMenu.scrollHeight + 'px';
            } else {
                dropdownMenu.style.maxHeight = '0';
            }
        });
    });
}

/**
 * Initialize stats counters with animation
 */
function initStats() {
    const statsElements = document.querySelectorAll('.stats-card h2');
    
    statsElements.forEach(element => {
        const target = parseInt(element.textContent.replace(/,/g, ''));
        const duration = 1500; // ms
        const step = 30; // ms
        const increment = target / (duration / step);
        let current = 0;
        
        const counter = setInterval(() => {
            current += increment;
            if (current >= target) {
                current = target;
                clearInterval(counter);
            }
            element.textContent = Math.floor(current).toLocaleString();
        }, step);
    });
}

/**
 * Initialize dashboard charts
 */
function initCharts() {
    initBookingsChart();
    initServicesChart();
    initRevenueChart();
    initUserChart();
}

/**
 * Initialize bookings chart
 */
function initBookingsChart() {
    const bookingsChartEl = document.getElementById('bookingsChart');
    if (!bookingsChartEl) return;

    // Sample data for demonstration
    const monthlyLabels = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
    const weeklyLabels = ['Week 1', 'Week 2', 'Week 3', 'Week 4'];
    
    const bookingsData = {
        monthly: [12, 19, 15, 25, 22, 30, 28, 25, 35, 32, 40, 45],
        weekly: [18, 25, 30, 22]
    };
    
    const revenueData = {
        monthly: [35000, 42000, 38000, 50000, 48000, 60000, 55000, 58000, 70000, 72000, 80000, 85000],
        weekly: [42000, 55000, 65000, 48000]
    };
    
    // Default chart type
    let chartType = 'monthly';
    let chartLabels = monthlyLabels;
    
    // Create chart
    const ctx = bookingsChartEl.getContext('2d');
    const bookingsChart = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: chartLabels,
            datasets: [
                {
                    label: 'Bookings',
                    data: bookingsData[chartType],
                    backgroundColor: 'rgba(52, 152, 219, 0.5)',
                    borderColor: 'rgba(52, 152, 219, 1)',
                    borderWidth: 1,
                    yAxisID: 'y',
                    order: 2
                },
                {
                    label: 'Revenue (FCFA)',
                    data: revenueData[chartType],
                    backgroundColor: 'rgba(46, 204, 113, 0)',
                    borderColor: 'rgba(46, 204, 113, 1)',
                    borderWidth: 2,
                    type: 'line',
                    yAxisID: 'y1',
                    order: 1
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Number of Bookings'
                    }
                },
                y1: {
                    beginAtZero: true,
                    position: 'right',
                    grid: {
                        drawOnChartArea: false
                    },
                    title: {
                        display: true,
                        text: 'Revenue (FCFA)'
                    }
                }
            },
            plugins: {
                tooltip: {
                    mode: 'index',
                    intersect: false
                },
                legend: {
                    position: 'top'
                }
            }
        }
    });
    
    // Handle chart type changes
    const chartTypeButtons = document.querySelectorAll('.chart-type');
    chartTypeButtons.forEach(button => {
        button.addEventListener('click', function() {
            chartType = this.getAttribute('data-chart-type');
            chartLabels = chartType === 'monthly' ? monthlyLabels : weeklyLabels;
            
            // Update chart data
            bookingsChart.data.labels = chartLabels;
            bookingsChart.data.datasets[0].data = bookingsData[chartType];
            bookingsChart.data.datasets[1].data = revenueData[chartType];
            bookingsChart.update();
        });
    });
}

/**
 * Initialize services distribution chart
 */
function initServicesChart() {
    const servicesChartEl = document.getElementById('servicesChart');
    if (!servicesChartEl) return;
    
    // Sample data for demonstration
    const data = {
        labels: ['Talk Show', 'Music Video', 'Interview', 'Podcast', 'Live Streaming'],
        datasets: [{
            label: 'Bookings by Service',
            data: [25, 35, 20, 15, 30],
            backgroundColor: [
                'rgba(52, 152, 219, 0.7)',
                'rgba(46, 204, 113, 0.7)',
                'rgba(155, 89, 182, 0.7)',
                'rgba(241, 196, 15, 0.7)',
                'rgba(231, 76, 60, 0.7)'
            ],
            borderColor: [
                'rgba(52, 152, 219, 1)',
                'rgba(46, 204, 113, 1)',
                'rgba(155, 89, 182, 1)',
                'rgba(241, 196, 15, 1)',
                'rgba(231, 76, 60, 1)'
            ],
            borderWidth: 1
        }]
    };
    
    // Create chart
    const ctx = servicesChartEl.getContext('2d');
    const servicesChart = new Chart(ctx, {
        type: 'doughnut',
        data: data,
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom'
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            const label = context.label || '';
                            const value = context.raw || 0;
                            const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
                            const percentage = Math.round((value / total) * 100);
                            return `${label}: ${value} (${percentage}%)`;
                        }
                    }
                }
            }
        }
    });
}

/**
 * Initialize revenue chart for the revenue section
 */
function initRevenueChart() {
    const revenueChartEl = document.getElementById('revenueChart');
    if (!revenueChartEl) return;
    
    // Sample data for demonstration
    const data = {
        labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
        datasets: [{
            label: 'Revenue',
            data: [35000, 42000, 38000, 50000, 48000, 60000, 55000, 58000, 70000, 72000, 80000, 85000],
            backgroundColor: 'rgba(46, 204, 113, 0.2)',
            borderColor: 'rgba(46, 204, 113, 1)',
            borderWidth: 2,
            fill: true,
            tension: 0.4
        }]
    };
    
    // Create chart
    const ctx = revenueChartEl.getContext('2d');
    const revenueChart = new Chart(ctx, {
        type: 'line',
        data: data,
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Revenue (FCFA)'
                    }
                }
            },
            plugins: {
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return `Revenue: ${context.raw.toLocaleString()} FCFA`;
                        }
                    }
                }
            }
        }
    });
}

/**
 * Initialize user statistics chart
 */
function initUserChart() {
    const userChartEl = document.getElementById('userChart');
    if (!userChartEl) return;
    
    // Sample data for demonstration
    const data = {
        labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
        datasets: [{
            label: 'New Users',
            data: [15, 20, 18, 25, 22, 30, 28, 32, 35, 40, 38, 45],
            backgroundColor: 'rgba(155, 89, 182, 0.2)',
            borderColor: 'rgba(155, 89, 182, 1)',
            borderWidth: 2,
            fill: true,
            tension: 0.4
        }]
    };
    
    // Create chart
    const ctx = userChartEl.getContext('2d');
    const userChart = new Chart(ctx, {
        type: 'line',
        data: data,
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Number of Users'
                    }
                }
            }
        }
    });
}

/**
 * Initialize the booking calendar
 */
function initCalendar() {
    const calendarEl = document.getElementById('dashboard-calendar');
    if (!calendarEl) return;
    
    // Check if FullCalendar is available
    if (typeof FullCalendar === 'undefined') {
        calendarEl.innerHTML = '<div class="alert alert-warning">FullCalendar library not loaded. Please include FullCalendar in your project.</div>';
        return;
    }
    
    // Sample events for demonstration
    const events = [
        {
            title: 'Talk Show - John Smith',
            start: '2023-04-15T10:00:00',
            end: '2023-04-15T12:00:00',
            color: '#3498db'
        },
        {
            title: 'Music Video - Harmony Band',
            start: '2023-04-17T14:00:00',
            end: '2023-04-17T18:00:00',
            color: '#2ecc71'
        },
        {
            title: 'Interview - Sarah Johnson',
            start: '2023-04-18T09:00:00',
            end: '2023-04-18T11:00:00',
            color: '#9b59b6'
        },
        {
            title: 'Live Streaming - Tech Conference',
            start: '2023-04-20T13:00:00',
            end: '2023-04-20T17:00:00',
            color: '#e74c3c'
        }
    ];
    
    // Initialize calendar
    const calendar = new FullCalendar.Calendar(calendarEl, {
        initialView: 'dayGridMonth',
        headerToolbar: {
            left: 'prev,next today',
            center: 'title',
            right: 'dayGridMonth,timeGridWeek,listWeek'
        },
        events: events,
        eventClick: function(info) {
            // Show booking details
            alert(`Booking Details:\n\nTitle: ${info.event.title}\nStart: ${info.event.start.toLocaleString()}\nEnd: ${info.event.end.toLocaleString()}`);
        },
        eventTimeFormat: {
            hour: '2-digit',
            minute: '2-digit',
            meridiem: 'short'
        },
        businessHours: {
            daysOfWeek: [1, 2, 3, 4, 5, 6], // Monday - Saturday
            startTime: '09:00',
            endTime: '19:00'
        }
    });
    
    calendar.render();
}

/**
 * Initialize DataTables for tables
 */
function initDataTables() {
    // Check if DataTables is available
    if (typeof $.fn.DataTable === 'undefined') {
        console.warn('DataTables library not loaded. Tables will use default styling.');
        return;
    }
    
    // Initialize DataTables for all tables with the 'datatable' class
    $('.datatable').DataTable({
        responsive: true,
        language: {
            search: "_INPUT_",
            searchPlaceholder: "Search...",
            lengthMenu: "Show _MENU_ entries",
            info: "Showing _START_ to _END_ of _TOTAL_ entries",
            infoEmpty: "Showing 0 to 0 of 0 entries",
            infoFiltered: "(filtered from _MAX_ total entries)",
            paginate: {
                first: '<i class="fas fa-angle-double-left"></i>',
                previous: '<i class="fas fa-angle-left"></i>',
                next: '<i class="fas fa-angle-right"></i>',
                last: '<i class="fas fa-angle-double-right"></i>'
            }
        }
    });
}

/**
 * Initialize event listeners for dashboard interactions
 */
function initEventListeners() {
    // Period selection for dashboard data
    const periodOptions = document.querySelectorAll('.period-option');
    if (periodOptions.length > 0) {
        periodOptions.forEach(option => {
            option.addEventListener('click', function(e) {
                e.preventDefault();
                
                // Update active state
                periodOptions.forEach(opt => opt.classList.remove('active'));
                this.classList.add('active');
                
                // Update dropdown button text
                const periodText = this.textContent;
                const periodDropdown = document.getElementById('periodDropdown');
                if (periodDropdown) {
                    periodDropdown.innerHTML = '<i class="fas fa-calendar me-1"></i> ' + periodText;
                }
                
                // Get period value and update dashboard
                const period = this.getAttribute('data-period');
                updateDashboardData(period);
            });
        });
    }
    
    // Handle delete confirmations
    const deleteButtons = document.querySelectorAll('.btn-delete');
    if (deleteButtons.length > 0) {
        deleteButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                
                const confirmMessage = this.getAttribute('data-confirm') || 'Are you sure you want to delete this item?';
                const targetUrl = this.getAttribute('href');
                
                if (confirm(confirmMessage)) {
                    window.location.href = targetUrl;
                }
            });
        });
    }
    
    // Handle form submissions with validation
    const adminForms = document.querySelectorAll('.admin-form');
    if (adminForms.length > 0) {
        adminForms.forEach(form => {
            form.addEventListener('submit', function(e) {
                const requiredFields = form.querySelectorAll('[required]');
                let isValid = true;
                
                requiredFields.forEach(field => {
                    if (!field.value.trim()) {
                        isValid = false;
                        field.classList.add('is-invalid');
                        
                        // Add error message if not exists
                        let errorMessage = field.nextElementSibling;
                        if (!errorMessage || !errorMessage.classList.contains('invalid-feedback')) {
                            errorMessage = document.createElement('div');
                            errorMessage.className = 'invalid-feedback';
                            errorMessage.textContent = 'This field is required';
                            field.parentNode.insertBefore(errorMessage, field.nextSibling);
                        }
                    } else {
                        field.classList.remove('is-invalid');
                    }
                });
                
                if (!isValid) {
                    e.preventDefault();
                }
            });
        });
    }
}

/**
 * Update dashboard data based on selected period
 * @param {string} period - The selected period (day, week, month, year, all)
 */
function updateDashboardData(period) {
    // This would typically be an AJAX call to fetch updated data
    console.log('Updating dashboard data for period:', period);
    
    // Show loading state
    const statsCards = document.querySelectorAll('.stats-card h2');
    statsCards.forEach(card => {
        card.innerHTML = '<span class="placeholder col-8"></span>';
    });
    
    // Sample AJAX call
    /*
    fetch('api/dashboard-data.php?period=' + period)
        .then(response => response.json())
        .then(data => {
            // Update stats cards
            document.querySelector('.stats-card.bookings h2').textContent = data.bookings.total.toLocaleString();
            document.querySelector('.stats-card.revenue h2').textContent = data.revenue.total.toLocaleString() + ' FCFA';
            document.querySelector('.stats-card.views h2').textContent = data.content.views.toLocaleString();
            document.querySelector('.stats-card.users h2').textContent = data.users.total.toLocaleString();
            
            // Update charts
            updateCharts(data);
            
            // Update tables
            updateTables(data);
        })
        .catch(error => {
            console.error('Error fetching dashboard data:', error);
            
            // Show error message
            const alertPlaceholder = document.getElementById('alertPlaceholder');
            if (alertPlaceholder) {
                alertPlaceholder.innerHTML = `
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <strong>Error!</strong> Failed to fetch dashboard data.
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                `;
            }
        });
    */
    
    // For demonstration, we'll simulate a response after a delay
    setTimeout(() => {
        // Sample data for each period
        const data = {
            day: {
                bookings: 5,
                revenue: 125000,
                views: 850,
                users: 15
            },
            week: {
                bookings: 25,
                revenue: 625000,
                views: 3500,
                users: 45
            },
            month: {
                bookings: 85,
                revenue: 2125000,
                views: 12500,
                users: 120
            },
            year: {
                bookings: 950,
                revenue: 23750000,
                views: 145000,
                users: 850
            },
            all: {
                bookings: 2450,
                revenue: 61250000,
                views: 320000,
                users: 1850
            }
        };
        
        // Get data for selected period
        const periodData = data[period] || data.month;
        
        // Update stats cards
        const bookingsStats = document.querySelector('.stats-card:nth-child(1) h2');
        const revenueStats = document.querySelector('.stats-card:nth-child(2) h2');
        const viewsStats = document.querySelector('.stats-card:nth-child(3) h2');
        const usersStats = document.querySelector('.stats-card:nth-child(4) h2');
        
        if (bookingsStats) bookingsStats.textContent = periodData.bookings.toLocaleString();
        if (revenueStats) revenueStats.textContent = periodData.revenue.toLocaleString() + ' FCFA';
        if (viewsStats) viewsStats.textContent = periodData.views.toLocaleString();
        if (usersStats) usersStats.textContent = periodData.users.toLocaleString();
    }, 500);
}

/**
 * Show success or error messages from URL parameters
 */
function showMessagesFromUrl() {
    const urlParams = new URLSearchParams(window.location.search);
    const successMessage = urlParams.get('success');
    const errorMessage = urlParams.get('error');
    
    const alertPlaceholder = document.getElementById('alertPlaceholder');
    if (!alertPlaceholder) return;
    
    if (successMessage) {
        alertPlaceholder.innerHTML = `
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                ${decodeURIComponent(successMessage)}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        `;
    } else if (errorMessage) {
        alertPlaceholder.innerHTML = `
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                ${decodeURIComponent(errorMessage)}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        `;
    }
    
    // Remove message from URL without refreshing the page
    if (successMessage || errorMessage) {
        const url = new URL(window.location);
        url.searchParams.delete('success');
        url.searchParams.delete('error');
        window.history.replaceState({}, '', url);
    }
}