/**
 * Admin Dashboard Charts
 * Initializes and manages charts for the admin dashboard
 */

// Chart color palette
const chartColors = {
    primary: '#4e73df',
    success: '#1cc88a',
    info: '#36b9cc',
    warning: '#f6c23e',
    danger: '#e74a3b',
    secondary: '#858796',
    light: '#f8f9fc',
    dark: '#5a5c69'
};

// Chart options for consistency
const commonChartOptions = {
    responsive: true,
    maintainAspectRatio: false,
    plugins: {
        legend: {
            position: 'top',
            labels: {
                usePointStyle: true,
                padding: 20
            }
        }
    }
};

// Global chart instances
let bookingsChart;
let servicesChart;

/**
 * Initialize admin dashboard charts
 */
function initAdminCharts() {
    try {
        // Initialize bookings & revenue chart
        initBookingsChart();
        
        // Initialize services distribution chart
        initServicesChart();
    } catch (error) {
        console.error('Error initializing charts:', error);
    }
}

/**
 * Initialize bookings & revenue chart
 */
function initBookingsChart() {
    try {
        const ctx = document.getElementById('bookingsChart');
        if (!ctx) return;
        
        // Sample data (should be replaced with real data from backend)
        const data = {
            labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
            datasets: [
                {
                    label: 'Bookings',
                    data: [15, 20, 25, 30, 28, 35, 40, 45, 50, 48, 52, 55],
                    backgroundColor: 'rgba(78, 115, 223, 0.2)',
                    borderColor: chartColors.primary,
                    borderWidth: 2,
                    pointBackgroundColor: chartColors.primary,
                    pointBorderColor: '#fff',
                    pointHoverBackgroundColor: '#fff',
                    pointHoverBorderColor: chartColors.primary,
                    tension: 0.3
                },
                {
                    label: 'Revenue',
                    data: [1500, 2000, 2500, 3000, 2800, 3500, 4000, 4500, 5000, 4800, 5200, 5500],
                    backgroundColor: 'rgba(28, 200, 138, 0.2)',
                    borderColor: chartColors.success,
                    borderWidth: 2,
                    pointBackgroundColor: chartColors.success,
                    pointBorderColor: '#fff',
                    pointHoverBackgroundColor: '#fff',
                    pointHoverBorderColor: chartColors.success,
                    tension: 0.3,
                    yAxisID: 'y1'
                }
            ]
        };
        
        // Chart options
        const options = {
            ...commonChartOptions,
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Bookings',
                        color: chartColors.primary
                    },
                    ticks: {
                        stepSize: 10
                    },
                    grid: {
                        drawBorder: false
                    }
                },
                y1: {
                    beginAtZero: true,
                    position: 'right',
                    title: {
                        display: true,
                        text: 'Revenue',
                        color: chartColors.success
                    },
                    grid: {
                        display: false
                    }
                },
                x: {
                    grid: {
                        drawBorder: false
                    }
                }
            }
        };
        
        // Create the chart
        bookingsChart = new Chart(ctx, {
            type: 'line',
            data: data,
            options: options
        });
    } catch (error) {
        console.error('Error initializing bookings chart:', error);
    }
}

/**
 * Initialize services distribution chart
 */
function initServicesChart() {
    try {
        const ctx = document.getElementById('servicesChart');
        if (!ctx) return;
        
        // Sample data (should be replaced with real data from backend)
        const data = {
            labels: ['Talk Show', 'Music Video', 'Interview', 'Podcast', 'Live Streaming'],
            datasets: [{
                data: [30, 25, 20, 15, 10],
                backgroundColor: [
                    chartColors.primary,
                    chartColors.success,
                    chartColors.info,
                    chartColors.warning,
                    chartColors.danger
                ],
                borderColor: '#ffffff',
                borderWidth: 2
            }]
        };
        
        // Chart options
        const options = {
            ...commonChartOptions,
            cutout: '50%',
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        usePointStyle: true,
                        padding: 15
                    }
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            const label = context.label || '';
                            const value = context.raw || 0;
                            const total = context.dataset.data.reduce((a, b) => a + b, 0);
                            const percentage = ((value / total) * 100).toFixed(1);
                            return `${label}: ${value} (${percentage}%)`;
                        }
                    }
                }
            }
        };
        
        // Create the chart
        servicesChart = new Chart(ctx, {
            type: 'doughnut',
            data: data,
            options: options
        });
    } catch (error) {
        console.error('Error initializing services chart:', error);
    }
}

/**
 * Update charts with new data
 * @param {Object} data The new data
 */
function updateCharts(data) {
    try {
        // Update bookings chart
        if (bookingsChart && data.bookings_chart) {
            bookingsChart.data.labels = data.bookings_chart.labels;
            bookingsChart.data.datasets[0].data = data.bookings_chart.bookings;
            bookingsChart.data.datasets[1].data = data.bookings_chart.revenue;
            bookingsChart.update();
        }
        
        // Update services chart
        if (servicesChart && data.services_chart) {
            servicesChart.data.labels = data.services_chart.labels;
            servicesChart.data.datasets[0].data = data.services_chart.data;
            servicesChart.update();
        }
    } catch (error) {
        console.error('Error updating charts:', error);
    }
}