/**
 * Admin Dashboard Calendar
 * Initializes and manages the booking calendar for the admin dashboard
 */

// Global calendar instance
let adminCalendar;

/**
 * Initialize admin dashboard calendar
 */
function initAdminCalendar() {
    try {
        const calendarEl = document.getElementById('dashboard-calendar');
        if (!calendarEl) return;
        
        adminCalendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,timeGridDay'
            },
            themeSystem: 'bootstrap5',
            selectable: true,
            selectMirror: true,
            dayMaxEvents: true,
            eventTimeFormat: {
                hour: '2-digit',
                minute: '2-digit',
                meridiem: 'short'
            },
            events: fetchCalendarEvents,
            eventClick: handleEventClick,
            dateClick: handleDateClick,
            eventColor: '#4e73df',
            eventBorderColor: '#3a5ccc',
            buttonText: {
                today: 'Today',
                month: 'Month',
                week: 'Week',
                day: 'Day'
            },
            height: 'auto',
            firstDay: 1 // Start from Monday
        });
        
        adminCalendar.render();
        
        // Adjust calendar size on window resize
        window.addEventListener('resize', () => {
            adminCalendar.updateSize();
        });
        
        // Handle events filter (if exists)
        const filterSelector = document.getElementById('calendar-filter');
        if (filterSelector) {
            filterSelector.addEventListener('change', function() {
                adminCalendar.refetchEvents();
            });
        }
    } catch (error) {
        console.error('Error initializing calendar:', error);
    }
}

/**
 * Fetch calendar events from backend
 * @param {Object} info Information object containing start and end dates
 * @param {Function} successCallback Callback for successful fetch
 * @param {Function} failureCallback Callback for fetch failure
 */
function fetchCalendarEvents(info, successCallback, failureCallback) {
    try {
        // Sample data (should be replaced with actual AJAX call)
        const events = [
            {
                title: 'Talk Show - Client A',
                start: '2025-04-15T10:00:00',
                end: '2025-04-15T12:00:00',
                id: '1',
                extendedProps: {
                    status: 'confirmed',
                    client: 'Client A',
                    service: 'Talk Show'
                },
                className: 'event-confirmed'
            },
            {
                title: 'Music Video - Client B',
                start: '2025-04-17T14:00:00',
                end: '2025-04-17T18:00:00',
                id: '2',
                extendedProps: {
                    status: 'pending',
                    client: 'Client B',
                    service: 'Music Video'
                },
                className: 'event-pending'
            },
            {
                title: 'Podcast - Client C',
                start: '2025-04-21T09:00:00',
                end: '2025-04-21T11:00:00',
                id: '3',
                extendedProps: {
                    status: 'confirmed',
                    client: 'Client C',
                    service: 'Podcast'
                },
                className: 'event-confirmed'
            },
            {
                title: 'Interview - Client D',
                start: '2025-04-23T13:00:00',
                end: '2025-04-23T15:00:00',
                id: '4',
                extendedProps: {
                    status: 'completed',
                    client: 'Client D',
                    service: 'Interview'
                },
                className: 'event-completed'
            }
        ];
        
        // In a real implementation, fetch events from server
        /*
        fetch('admin/api/calendar-events.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                start: info.startStr,
                end: info.endStr,
                filter: document.getElementById('calendar-filter')?.value || 'all'
            })
        })
        .then(response => response.json())
        .then(data => {
            successCallback(data.events);
        })
        .catch(error => {
            console.error('Error fetching calendar events:', error);
            failureCallback(error);
        });
        */
        
        // For demo, use sample data
        successCallback(events);
    } catch (error) {
        console.error('Error fetching calendar events:', error);
        failureCallback(error);
    }
}

/**
 * Handle event click
 * @param {Object} info Event information
 */
function handleEventClick(info) {
    try {
        const event = info.event;
        const id = event.id;
        
        // Redirect to booking details page
        window.location.href = `booking-details.php?id=${id}`;
    } catch (error) {
        console.error('Error handling event click:', error);
    }
}

/**
 * Handle date click
 * @param {Object} info Date information
 */
function handleDateClick(info) {
    try {
        // Redirect to booking creation page with pre-selected date
        window.location.href = `add-booking.php?date=${info.dateStr}`;
    } catch (error) {
        console.error('Error handling date click:', error);
    }
}

/**
 * Refresh calendar events
 */
function refreshCalendarEvents() {
    if (adminCalendar) {
        adminCalendar.refetchEvents();
    }
}