<?php
/**
 * Admin Dashboard
 * Main entry point for the admin panel
 */

// Include configuration
require_once '../includes/config.php';

// Include required classes
require_once '../includes/db.php';
require_once '../includes/auth.php';
require_once '../includes/booking.php';
require_once '../includes/content.php';
require_once '../includes/payment.php';

// Initialize classes
$auth = new Auth();
$bookingManager = new Booking();
$contentManager = new Content();
$paymentManager = new Payment();

// Check if user is logged in and is admin
if (!$auth->isLoggedIn() || !$auth->isAdmin()) {
    // Redirect to login page
    header('Location: login.php');
    exit;
}

// Get current user
$currentUser = $auth->getCurrentUser();

// Load language file
$lang = isset($_SESSION['language']) ? $_SESSION['language'] : DEFAULT_LANGUAGE;
require_once "../languages/{$lang}.php";

// Get dashboard statistics
$bookingStats = $bookingManager->getBookingStats('month');
$paymentStats = $paymentManager->getPaymentStats('month');
$contentStats = $contentManager->getContentStats();
$userStats = $auth->getUserStats();

// Get recent bookings
$db = Database::getInstance();
$recentBookings = $db->getRows(
    "SELECT b.*, u.name as client_name, s.name as service_name 
     FROM bookings b 
     LEFT JOIN users u ON b.user_id = u.id 
     LEFT JOIN services s ON b.service_id = s.id
     ORDER BY b.created_at DESC LIMIT 5"
);

// Get recent payments
$recentPayments = $paymentManager->getRecentPayments(5);

// Set page title
$pageTitle = $lang_data['admin_dashboard'] . ' - ' . SITE_NAME;

// Include admin header
include 'includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <?php include 'includes/sidebar.php'; ?>
        
        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2"><?php echo $lang_data['dashboard']; ?></h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <div class="btn-group me-2">
                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="window.print()">
                            <i class="fas fa-print me-1"></i> <?php echo $lang_data['print']; ?>
                        </button>
                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="exportDashboard()">
                            <i class="fas fa-download me-1"></i> <?php echo $lang_data['export']; ?>
                        </button>
                    </div>
                    <div class="dropdown">
                        <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button" id="periodDropdown" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="fas fa-calendar me-1"></i> <?php echo $lang_data['this_month']; ?>
                        </button>
                        <ul class="dropdown-menu" aria-labelledby="periodDropdown">
                            <li><a class="dropdown-item period-option" href="#" data-period="day"><?php echo $lang_data['today']; ?></a></li>
                            <li><a class="dropdown-item period-option" href="#" data-period="week"><?php echo $lang_data['this_week']; ?></a></li>
                            <li><a class="dropdown-item period-option active" href="#" data-period="month"><?php echo $lang_data['this_month']; ?></a></li>
                            <li><a class="dropdown-item period-option" href="#" data-period="year"><?php echo $lang_data['this_year']; ?></a></li>
                            <li><a class="dropdown-item period-option" href="#" data-period="all"><?php echo $lang_data['all_time']; ?></a></li>
                        </ul>
                    </div>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="row">
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card stats-card h-100">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="card-title text-muted mb-0"><?php echo $lang_data['total_bookings']; ?></h6>
                                    <h2 class="mt-2 mb-0"><?php echo number_format($bookingStats['total']); ?></h2>
                                    <p class="text-success mb-0"><i class="fas fa-arrow-up"></i> 12% <?php echo $lang_data['since_last_month']; ?></p>
                                </div>
                                <div class="stats-icon bg-primary">
                                    <i class="fas fa-calendar-check"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card stats-card h-100">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="card-title text-muted mb-0"><?php echo $lang_data['revenue']; ?></h6>
                                    <h2 class="mt-2 mb-0"><?php echo number_format($paymentStats['total_revenue']) . ' ' . CURRENCY; ?></h2>
                                    <p class="text-success mb-0"><i class="fas fa-arrow-up"></i> 8% <?php echo $lang_data['since_last_month']; ?></p>
                                </div>
                                <div class="stats-icon bg-success">
                                    <i class="fas fa-money-bill-wave"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card stats-card h-100">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="card-title text-muted mb-0"><?php echo $lang_data['content_views']; ?></h6>
                                    <h2 class="mt-2 mb-0"><?php echo isset($contentStats['total_views']) ? number_format($contentStats['total_views']) : '0'; ?></h2>
                                    <p class="text-success mb-0"><i class="fas fa-arrow-up"></i> 18% <?php echo $lang_data['since_last_month']; ?></p>
                                </div>
                                <div class="stats-icon bg-info">
                                    <i class="fas fa-eye"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card stats-card h-100">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h6 class="card-title text-muted mb-0"><?php echo $lang_data['users']; ?></h6>
                                    <h2 class="mt-2 mb-0"><?php echo number_format($userStats['total']); ?></h2>
                                    <p class="text-success mb-0"><i class="fas fa-arrow-up"></i> 5% <?php echo $lang_data['since_last_month']; ?></p>
                                </div>
                                <div class="stats-icon bg-warning">
                                    <i class="fas fa-users"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Charts Row -->
            <div class="row">
                <div class="col-lg-8 mb-4">
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5 class="card-title mb-0"><?php echo $lang_data['bookings_revenue_overview']; ?></h5>
                            <div class="btn-group">
                                <button type="button" class="btn btn-sm btn-outline-secondary chart-type active" data-chart-type="monthly">
                                    <?php echo $lang_data['monthly']; ?>
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-secondary chart-type" data-chart-type="weekly">
                                    <?php echo $lang_data['weekly']; ?>
                                </button>
                            </div>
                        </div>
                        <div class="card-body">
                            <canvas id="bookingsChart" height="300"></canvas>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-4 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0"><?php echo $lang_data['service_distribution']; ?></h5>
                        </div>
                        <div class="card-body">
                            <canvas id="servicesChart" height="300"></canvas>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Recent Bookings & Activity Feed -->
            <div class="row">
                <div class="col-lg-7 mb-4">
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5 class="card-title mb-0"><?php echo $lang_data['recent_bookings']; ?></h5>
                            <a href="bookings.php" class="btn btn-sm btn-primary">
                                <?php echo $lang_data['view_all']; ?>
                            </a>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-hover align-middle">
                                    <thead>
                                        <tr>
                                            <th><?php echo $lang_data['reference']; ?></th>
                                            <th><?php echo $lang_data['client']; ?></th>
                                            <th><?php echo $lang_data['service']; ?></th>
                                            <th><?php echo $lang_data['date_time']; ?></th>
                                            <th><?php echo $lang_data['status']; ?></th>
                                            <th><?php echo $lang_data['actions']; ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($recentBookings)): ?>
                                            <tr>
                                                <td colspan="6" class="text-center"><?php echo $lang_data['no_bookings']; ?></td>
                                            </tr>
                                        <?php else: ?>
                                            <?php foreach ($recentBookings as $booking): ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($booking['booking_reference']); ?></td>
                                                    <td><?php echo htmlspecialchars($booking['client_name']); ?></td>
                                                    <td><?php echo htmlspecialchars($booking['service_name']); ?></td>
                                                    <td>
                                                        <?php 
                                                        echo date('M j, Y', strtotime($booking['booking_date'])) . '<br>' . 
                                                             date('g:i A', strtotime($booking['start_time'])) . ' - ' . 
                                                             date('g:i A', strtotime($booking['end_time']));
                                                        ?>
                                                    </td>
                                                    <td>
                                                        <?php
                                                        $statusClass = '';
                                                        switch ($booking['status']) {
                                                            case 'pending':
                                                                $statusClass = 'bg-warning';
                                                                break;
                                                            case 'confirmed':
                                                                $statusClass = 'bg-success';
                                                                break;
                                                            case 'completed':
                                                                $statusClass = 'bg-info';
                                                                break;
                                                            case 'canceled':
                                                                $statusClass = 'bg-danger';
                                                                break;
                                                        }
                                                        ?>
                                                        <span class="badge <?php echo $statusClass; ?>">
                                                            <?php echo $lang_data[$booking['status']]; ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <a href="booking-details.php?id=<?php echo $booking['id']; ?>" class="btn btn-sm btn-outline-primary">
                                                            <i class="fas fa-eye"></i>
                                                        </a>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-5 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0"><?php echo $lang_data['recent_activities']; ?></h5>
                        </div>
                        <div class="card-body">
                            <div class="activity-feed">
                                <?php
                                // Combine recent bookings and payments to create activity feed
                                $activities = [];
                                
                                // Add bookings to activities
                                foreach ($recentBookings as $booking) {
                                    $activities[] = [
                                        'type' => 'booking',
                                        'icon' => 'calendar-check',
                                        'icon_class' => 'bg-primary',
                                        'title' => $lang_data['new_booking_from'] . ' ' . $booking['client_name'],
                                        'description' => $booking['service_name'],
                                        'time' => $booking['created_at']
                                    ];
                                }
                                
                                // Add payments to activities
                                foreach ($recentPayments as $payment) {
                                    $activities[] = [
                                        'type' => 'payment',
                                        'icon' => 'money-bill-wave',
                                        'icon_class' => 'bg-success',
                                        'title' => $lang_data['payment_received'] . ' ' . $payment['user_name'],
                                        'description' => number_format($payment['amount']) . ' ' . CURRENCY,
                                        'time' => $payment['created_at']
                                    ];
                                }
                                
                                // Sort activities by time (newest first)
                                usort($activities, function($a, $b) {
                                    return strtotime($b['time']) - strtotime($a['time']);
                                });
                                
                                // Display activities
                                if (empty($activities)):
                                ?>
                                    <div class="text-center py-5">
                                        <p class="text-muted"><?php echo $lang_data['no_recent_activities']; ?></p>
                                    </div>
                                <?php else: ?>
                                    <?php foreach (array_slice($activities, 0, 10) as $activity): ?>
                                        <div class="activity-item">
                                            <div class="activity-icon <?php echo $activity['icon_class']; ?>">
                                                <i class="fas fa-<?php echo $activity['icon']; ?>"></i>
                                            </div>
                                            <div class="activity-content">
                                                <p class="mb-0"><?php echo htmlspecialchars($activity['title']); ?></p>
                                                <p class="small mb-0"><?php echo htmlspecialchars($activity['description']); ?></p>
                                                <small class="text-muted">
                                                    <?php 
                                                    // Format time as "X time ago"
                                                    $timeAgo = time() - strtotime($activity['time']);
                                                    if ($timeAgo < 60) {
                                                        echo $timeAgo . ' ' . $lang_data['seconds_ago'];
                                                    } elseif ($timeAgo < 3600) {
                                                        echo floor($timeAgo / 60) . ' ' . $lang_data['minutes_ago'];
                                                    } elseif ($timeAgo < 86400) {
                                                        echo floor($timeAgo / 3600) . ' ' . $lang_data['hours_ago'];
                                                    } else {
                                                        echo floor($timeAgo / 86400) . ' ' . $lang_data['days_ago'];
                                                    }
                                                    ?>
                                                </small>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Calendar & Popular Content -->
            <div class="row">
                <div class="col-lg-8 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0"><?php echo $lang_data['booking_calendar']; ?></h5>
                        </div>
                        <div class="card-body">
                            <div id="dashboard-calendar"></div>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-4 mb-4">
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5 class="card-title mb-0"><?php echo $lang_data['popular_content']; ?></h5>
                            <a href="content.php" class="btn btn-sm btn-primary">
                                <?php echo $lang_data['view_all']; ?>
                            </a>
                        </div>
                        <div class="card-body">
                            <?php if (empty($contentStats['popular'])): ?>
                                <div class="text-center py-5">
                                    <p class="text-muted"><?php echo $lang_data['no_content']; ?></p>
                                </div>
                            <?php else: ?>
                                <div class="list-group">
                                    <?php foreach ($contentStats['popular'] as $content): ?>
                                        <a href="content-details.php?id=<?php echo $content['id']; ?>" class="list-group-item list-group-item-action">
                                            <div class="d-flex w-100 justify-content-between align-items-center">
                                                <div class="d-flex align-items-center">
                                                    <img src="<?php echo htmlspecialchars($content['thumbnail']); ?>" 
                                                         alt="<?php echo htmlspecialchars($content['title']); ?>" 
                                                         class="rounded me-3" style="width: 50px; height: 50px; object-fit: cover;">
                                                    <div>
                                                        <h6 class="mb-1"><?php echo htmlspecialchars($content['title']); ?></h6>
                                                        <small class="text-muted">
                                                            <i class="fas fa-eye me-1"></i> <?php echo number_format($content['view_count']); ?> <?php echo $lang_data['views']; ?>
                                                        </small>
                                                    </div>
                                                </div>
                                            </div>
                                        </a>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<?php include 'includes/footer.php'; ?>

<!-- Initialize charts and calendar -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize charts
    initAdminCharts();
    
    // Initialize calendar
    initAdminCalendar();
    
    // Period selection
    document.querySelectorAll('.period-option').forEach(function(option) {
        option.addEventListener('click', function(e) {
            e.preventDefault();
            
            // Remove active class from all options
            document.querySelectorAll('.period-option').forEach(function(opt) {
                opt.classList.remove('active');
            });
            
            // Add active class to selected option
            this.classList.add('active');
            
            // Update dropdown button text
            const period = this.getAttribute('data-period');
            document.getElementById('periodDropdown').textContent = this.textContent;
            
            // Update dashboard data
            updateDashboardData(period);
        });
    });
    
    // Chart type selection
    document.querySelectorAll('.chart-type').forEach(function(option) {
        option.addEventListener('click', function(e) {
            e.preventDefault();
            
            // Remove active class from all options
            document.querySelectorAll('.chart-type').forEach(function(opt) {
                opt.classList.remove('active');
            });
            
            // Add active class to selected option
            this.classList.add('active');
            
            // Update chart type
            const chartType = this.getAttribute('data-chart-type');
            updateChartType(chartType);
        });
    });
});

// Function to update dashboard data
function updateDashboardData(period) {
    // In a real application, this would make an AJAX call to fetch updated data
    console.log('Updating dashboard data for period:', period);
    
    // Example AJAX call
    /*
    fetch('admin/api/dashboard-data.php?period=' + period)
        .then(response => response.json())
        .then(data => {
            // Update stats cards
            document.querySelector('#bookings-stats').textContent = data.bookings.total;
            document.querySelector('#revenue-stats').textContent = data.payments.total_revenue;
            document.querySelector('#views-stats').textContent = data.content.total_views;
            document.querySelector('#users-stats').textContent = data.users.total;
            
            // Update charts
            updateCharts(data);
        })
        .catch(error => console.error('Error fetching dashboard data:', error));
    */
}

// Function to update chart type
function updateChartType(chartType) {
    // In a real application, this would update the chart data and display
    console.log('Updating chart type to:', chartType);
    
    // Example chart update
    /*
    if (chartType === 'monthly') {
        bookingsChart.data.labels = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
        // Update data...
    } else {
        bookingsChart.data.labels = ['Week 1', 'Week 2', 'Week 3', 'Week 4'];
        // Update data...
    }
    bookingsChart.update();
    */
}

// Function to export dashboard data
function exportDashboard() {
    // In a real application, this would generate a report file
    console.log('Exporting dashboard data');
    alert('<?php echo $lang_data['export_started']; ?>');
    
    // Example export
    /*
    window.location.href = 'admin/export-dashboard.php?format=pdf';
    */
}
</script>
